(* ========================================================================= *)
(* CHESS                                                                     *)
(* Copyright (c) 2004 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature Chess =
sig

(* ------------------------------------------------------------------------- *)
(* Constants.                                                                *)
(* ------------------------------------------------------------------------- *)

val FILES : int

val RANKS : int

val IMAGE_PATH : string ref

val PGN_ESCAPE : (string -> unit) ref

(* ------------------------------------------------------------------------- *)
(* Pieces.                                                                   *)
(* ------------------------------------------------------------------------- *)

datatype piece =
    King
  | Queen
  | Rook
  | Bishop
  | Knight
  | Pawn

val all_pieces : piece list

val piece_compare : piece * piece -> order

val piece_to_char : piece -> char
val char_to_piece : char -> piece

val piece_to_string : piece -> string
val string_to_piece : string -> piece

val side_piece_to_char : Side.side * piece -> char
val char_to_side_piece : char -> Side.side * piece

val side_piece_to_string : Side.side * piece -> string
val string_to_side_piece : string -> Side.side * piece

(* ------------------------------------------------------------------------- *)
(* Squares.                                                                  *)
(* ------------------------------------------------------------------------- *)

datatype square = Square of {rank : int, file : int}

val all_files : int list

val all_ranks : int list

val file_to_char : int -> char
val char_to_file : char -> int

val rank_to_char : int -> char
val char_to_rank : char -> int

val file_to_string : int -> string
val string_to_file : string -> int

val rank_to_string : int -> string
val string_to_rank : string -> int

val square_compare : square * square -> order

val all_squares : square list

val square_to_string : square -> string
val string_to_square : string -> square

(* ------------------------------------------------------------------------- *)
(* Diagrams.                                                                 *)
(* ------------------------------------------------------------------------- *)

datatype diagram = Diagram of square -> (Side.side * piece) option

val diagram_equal : diagram * diagram -> bool

val diagram_size : diagram -> int

val flip_diagram : diagram -> diagram

val fold_pieces :
    (square * (Side.side * piece) * 'a -> 'a) -> 'a -> diagram -> 'a

val diagram_to_string : diagram -> string

val diagram_to_fen_prefix : diagram -> string

val fen_prefix_to_diagram : string -> diagram

val diagram_to_image : diagram -> Image.image

(* ------------------------------------------------------------------------- *)
(* Castling.                                                                 *)
(* ------------------------------------------------------------------------- *)

datatype castling_side = Kingside | Queenside

type castling_rights = castling_side -> bool

type castling = Side.side -> castling_rights

val all_castling_sides : castling_side list

val castling_rights_equal : castling_rights * castling_rights -> bool

val castling_equal : castling * castling -> bool

val no_castling : castling

val castling_to_string : castling -> string

val string_to_castling : string -> castling

(* ------------------------------------------------------------------------- *)
(* Positions.                                                                *)
(* ------------------------------------------------------------------------- *)

datatype position =
    Position of
      {diagram : diagram,
       to_move : Side.side,
       castling : castling,
       en_passant : square option,
       halfmove_clock : int,
       fullmove_number : int}

val initial_position : position

val position_equal : position * position -> bool

val flip_position : position -> position

val legal_position : position -> bool

val check_position : position -> bool

val checkmate_position : position -> bool

val stalemate_position : position -> bool

val position_to_string : position -> string

val position_to_image : position -> Image.image

val fen_to_position : string -> position

val position_to_fen : position -> string

(* ------------------------------------------------------------------------- *)
(* Moves.                                                                    *)
(* ------------------------------------------------------------------------- *)

datatype move =
    Move of {piece : piece, from : square, to : square}
  | Promotion of {from : square, to : square, promotion : piece}
  | En_passant of {from : square, to : square}
  | Castling of castling_side

val san_to_move : position -> string -> move

val move_to_san : position -> move -> string

val all_legal_moves : position -> (move * position) list

val apply_move : move -> position -> position

val deduce_move : position -> position -> move

(* ------------------------------------------------------------------------- *)
(* Games.                                                                    *)
(* ------------------------------------------------------------------------- *)

type tag = {name : string, value : string}

type halfmove = {position : position, comment : string}

datatype result =
    White_wins
  | Black_wins
  | Draw

datatype game =
    Game of
      {tags : tag list,
       halfmoves : halfmove list,
       result : result option}

val game_result : game -> result option

val game_length : game -> int

val fens_to_game : string list -> game

val pgn_date : Date.date -> string

val pgn_time : Date.date -> string

val game_to_pgn : game -> string

val pgn_to_game : string -> game

val games_to_pgn : {filename : string} -> game list -> unit

val pgn_to_games : {filename : string} -> game list

val game_to_html : {path : string, basename : string} -> game -> Html.block

val pgn_to_html : {path : string, basename : string} -> unit

end
