(* ========================================================================= *)
(* CHESS                                                                     *)
(* Copyright (c) 2004 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Chess :> Chess =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Constants.                                                                *)
(* ------------------------------------------------------------------------- *)

val FILES = 8 and RANKS = 8;

val SQUARE_LENGTH = 48 and SQUARE_OVERLAP = 0;

val STRIPE_PERIOD = 4 and STRIPE_COLOUR = Image.black;

val FRAME_WIDTH = 5 and FRAME_SPACE = 0;

val BORDER_WIDTH = 2.0;

val IMAGE_PATH = ref ".";

val PGN_ESCAPE : (string -> unit) ref = ref (K ());

(* ------------------------------------------------------------------------- *)
(* Helper functions.                                                         *)
(* ------------------------------------------------------------------------- *)

fun string_to_char err s =
    case String.explode s of [c] => c | _ => raise Error err;

fun int_to_char err n = string_to_char err (Int.toString n);

fun natFromString err s =
    case Int.fromString s of
      SOME i => i
    | NONE => raise Error err;

val input_chars = Stream.fromList o String.explode;

(* ------------------------------------------------------------------------- *)
(* Pieces.                                                                   *)
(* ------------------------------------------------------------------------- *)

datatype piece =
    King
  | Queen
  | Rook
  | Bishop
  | Knight
  | Pawn;

val all_pieces = [King,Queen,Rook,Bishop,Knight,Pawn];

fun piece_compare (King,King) = EQUAL
  | piece_compare (King,_) = LESS
  | piece_compare (_,King) = GREATER
  | piece_compare (Queen,Queen) = EQUAL
  | piece_compare (Queen,_) = LESS
  | piece_compare (_,Queen) = GREATER
  | piece_compare (Rook,Rook) = EQUAL
  | piece_compare (Rook,_) = LESS
  | piece_compare (_,Rook) = GREATER
  | piece_compare (Bishop,Bishop) = EQUAL
  | piece_compare (Bishop,_) = LESS
  | piece_compare (_,Bishop) = GREATER
  | piece_compare (Knight,Knight) = EQUAL
  | piece_compare (Knight,_) = LESS
  | piece_compare (_,Knight) = GREATER
  | piece_compare (Pawn,Pawn) = EQUAL;

fun piece_to_char King = #"K"
  | piece_to_char Queen = #"Q"
  | piece_to_char Rook = #"R"
  | piece_to_char Bishop = #"B"
  | piece_to_char Knight = #"N"
  | piece_to_char Pawn = #"P";

fun char_to_piece #"K" = King
  | char_to_piece #"Q" = Queen
  | char_to_piece #"R" = Rook
  | char_to_piece #"B" = Bishop
  | char_to_piece #"N" = Knight
  | char_to_piece #"P" = Pawn
  | char_to_piece _ = raise Error "char_to_piece";

val piece_to_string = str o piece_to_char;

val string_to_piece = char_to_piece o string_to_char "string_to_piece";

fun side_piece_to_char (Side.White,piece) = piece_to_char piece
  | side_piece_to_char (Side.Black,piece) = Char.toLower (piece_to_char piece);

fun char_to_side_piece c =
    let
      val side = if Char.isUpper c then Side.White else Side.Black
      val piece = char_to_piece (Char.toUpper c)
    in
      (side,piece)
    end;

val side_piece_to_string = str o side_piece_to_char;

val string_to_side_piece =
    char_to_side_piece o string_to_char "string_to_side_piece";

val pp_piece = Print.ppMap piece_to_string Print.ppString;

(* ------------------------------------------------------------------------- *)
(* Squares.                                                                  *)
(* ------------------------------------------------------------------------- *)

datatype square = Square of {file : int, rank : int};

val all_files = interval 1 FILES
and all_ranks = interval 1 RANKS;

val valid_file = fn file => 1 <= file andalso file <= FILES
and valid_rank = fn rank => 1 <= rank andalso rank <= RANKS;

fun dest_square (Square {file,rank}) = (file,rank);
val square_file = fst o dest_square
and square_rank = snd o dest_square;

fun mk_square (file,rank) = Square {file = file, rank = rank};

fun square_compare (sq1,sq2) =
    prodCompare Int.compare Int.compare (dest_square sq1, dest_square sq2);

fun on_board (Square {file,rank}) = valid_file file andalso valid_rank rank;

(* FEN order *)
val all_squares = List.map mk_square (cart all_files (List.rev all_ranks));

fun fold_squares f b = List.foldl f b all_squares;

fun queen_rook_square to_move =
    Square
      {file = 1,
       rank = case to_move of Side.White => 1 | Side.Black => RANKS};

fun king_square to_move =
    Square
      {file = 5,
       rank = case to_move of Side.White => 1 | Side.Black => RANKS};

fun king_rook_square to_move =
    Square
      {file = FILES,
       rank = case to_move of Side.White => 1 | Side.Black => RANKS};

fun flip_square (Square {file,rank}) =
    Square {file = file, rank = RANKS - (rank + 1)};

fun file_to_char file = chr (file + ord #"a" - 1);
fun rank_to_char rank = chr (rank + ord #"1" - 1);

val file_to_string = str o file_to_char
and rank_to_string = str o rank_to_char;

fun initial_pawn_rank Side.White = 2 | initial_pawn_rank Side.Black = RANKS - 1;

fun pawn_direction Side.White = 1 | pawn_direction Side.Black = ~1;

fun square_to_string (Square {file,rank}) =
    file_to_string file ^ rank_to_string rank;

local
  fun char_to_int start max exn c =
      let
        val n = ord c - ord start + 1
        val _ = (1 <= n andalso n <= max) orelse raise exn
      in
        n
      end;
in
  val char_to_file = char_to_int #"a" FILES (Error "char_to_file")
  and char_to_rank = char_to_int #"1" RANKS (Error "char_to_rank");
end;

val string_to_file = char_to_file o string_to_char "string_to_file"
and string_to_rank = char_to_rank o string_to_char "string_to_rank";

fun string_to_square s =
    case String.explode s of
      [f,r] => Square {file = char_to_file f, rank = char_to_rank r}
    | _ => raise Error "string_to_square";

val pp_square = Print.ppMap square_to_string Print.ppString;

(* ------------------------------------------------------------------------- *)
(* Diagrams                                                                  *)
(* ------------------------------------------------------------------------- *)

datatype diagram = Diagram of square -> (Side.side * piece) option;

fun diagram_equal (diagram1,diagram2) =
    let
      val Diagram on_square1 = diagram1
      and Diagram on_square2 = diagram2
    in
      List.all (fn sq => on_square1 sq = on_square2 sq) all_squares
    end;

fun fold_pieces f b diagram =
    let
(*Fen2ImgTrace3
      val () = trace "fold_pieces\n"
*)
      val Diagram on_square = diagram
      fun fold (sq,z) =
          let
(*Fen2ImgTrace4
            val () = Print.trace pp_square "sq" sq
*)
          in
            case on_square sq of
              NONE => z
            | SOME side_piece => f (sq,side_piece,z)
          end
    in
      fold_squares fold b
    end;

val diagram_size = fold_pieces (fn (_,_,n) => n + 1) 0;

fun flip_diagram diagram =
    let
      fun f (sq,(s,p),d) sq' =
          if sq' = flip_square sq then SOME (Side.opponent s, p) else d sq'
    in
      Diagram (fold_pieces f (K NONE) diagram)
    end;

fun diagram_to_string (Diagram diagram) =
    let
      fun rank r =
          let
            fun file f =
                case diagram (Square {file = f, rank = r}) of
                  NONE => ". "
                | SOME p => side_piece_to_string p ^ " "
          in
            List.map file all_files
          end

      val board = List.map rank (List.rev all_ranks)

      val frame = "+" ^ nChars #"-" (2 * FILES + 1) ^ "+" ^ "\n"
    in
      frame ^ "| " ^ join "|\n| " (List.map String.concat board) ^ "|\n" ^ frame
    end;

fun diagram_to_fen_prefix diagram =
    let
      val Diagram on_square = diagram
      type state = {rank : int, empty : int, result : char list}
      val initial : state = {rank = RANKS, empty = 0, result = []}
      fun norm (s as {empty = 0, ...}) = s
        | norm {rank,empty,result} =
          {rank = rank, empty = 0,
           result = int_to_char "diagram_to_fen" empty :: result}
      fun inc {rank,empty,result} : state =
          {rank = rank, empty = empty + 1, result = result}
      val finalize : state -> string = String.implode o List.rev o #result o norm
      fun advance (sq,state) =
          if square_rank sq <> #rank state then
            let
              val {rank = _, empty, result} = norm state
              val result = #"/" :: result
              val rank = square_rank sq
              val state = {rank = rank, empty = empty, result = result}
            in
              advance (sq,state)
            end
          else
            case on_square sq of
              NONE => inc state
            | SOME side_piece =>
              let
                val {rank,empty,result} = norm state
                val result = side_piece_to_char side_piece :: result
              in
                {rank = rank, empty = empty, result = result}
              end
    in
      finalize (List.foldl advance initial all_squares)
    end;

local
  infixr 9 >>++
  infixr 8 ++
  infixr 7 >>
  infixr 6 ||

  open Parse;

  val slash_parser = exactChar #"/";

  val digit_parser = some Char.isDigit >> (natFromString "digit_parser" o str);

  val number_parser =
      atLeastOne (some Char.isDigit) >>
      (natFromString "number_parser" o String.implode);

  val side_parser = maybe (total Side.fromChar);

  val side_piece_parser = maybe (total char_to_side_piece);
in
  fun diagram_parser input =
      let
        val array = Array2.array (RANKS,FILES,NONE)

        fun on_square (Square {rank,file}) =
            Array2.sub (array, rank - 1, file - 1)

        fun populate (rank,file) =
            let
              val _ = 1 <= rank orelse raise Error "too many ranks"
              val _ = file <= FILES + 1 orelse raise Error "too many files"
(*Fen2ImgTrace3
              val () = Print.trace Print.ppInt "rank" rank
              val () = Print.trace Print.ppInt "file" file
*)
              fun place p = Array2.update (array, rank - 1, file - 1, SOME p)
            in
              if file = FILES + 1 then
                slash_parser >> (K (rank - 1, 1))
              else
                digit_parser >> (fn n => (rank, file + n))
                || side_piece_parser >> (fn p => (place p; (rank, file + 1)))
            end

        val ((rank,file),input) = mmany populate (RANKS,1) input

        val _ = file = FILES + 1 orelse raise Error "ill-formed rank"
        val _ = rank = 1 orelse raise Error "not enough ranks"
      in
        (Diagram on_square, input)
      end;
end;

val fen_prefix_to_diagram = fst o diagram_parser o input_chars;

local
  fun king_attacks sq =
      let
        fun pred sq' = on_board sq' andalso sq <> sq'
        val Square {file,rank} = sq
        val attacked =
            let
              val files = [file - 1, file, file + 1]
              and ranks = [rank - 1, rank, rank + 1]
            in
              cart files ranks
            end
      in
        List.filter pred (List.map mk_square attacked)
      end;

  fun knight_attacks sq =
      let
        val Square {file,rank} = sq
        fun attacked file' rank' =
            let
              val files = [file - file', file + file']
              and ranks = [rank - rank', rank + rank']
            in
              List.filter on_board (List.map mk_square (cart files ranks))
            end
      in
        attacked 2 1 @ attacked 1 2
      end;

  fun line_attacks diagram sq vec =
      let
        val Diagram on_square = diagram
        val (file',rank') = vec
        val Square {file,rank} = sq
        val sq = Square {file = file + file', rank = rank + rank'}
      in
        if not (on_board sq) then []
        else if Option.isSome (on_square sq) then [sq]
        else sq :: line_attacks diagram sq vec
      end;

  fun rook_attacks diagram sq =
      List.concat (List.map (line_attacks diagram sq) [(~1,0),(1,0),(0,~1),(0,1)]);

  fun bishop_attacks diagram sq =
      List.concat (List.map (line_attacks diagram sq) [(~1,~1),(~1,1),(1,~1),(1,1)]);

  fun queen_attacks diagram sq =
      rook_attacks diagram sq @ bishop_attacks diagram sq;

  fun pawn_pushes side sq =
      let
        val Square {file,rank} = sq
        val rank = case side of Side.White => rank + 1 | Side.Black => rank - 1
      in
        List.filter on_board [mk_square (file,rank)]
      end;

  fun pawn_attacks side sq =
      let
        val Square {file,rank} = sq
        val rank = case side of Side.White => rank + 1 | Side.Black => rank - 1
        val files = [file - 1, file + 1]
      in
        List.filter on_board (List.map mk_square (cart files [rank]))
      end;

  fun piece_pushes _ (_,King) sq = king_attacks sq
    | piece_pushes _ (_,Knight) sq = knight_attacks sq
    | piece_pushes diagram (_,Rook) sq = rook_attacks diagram sq
    | piece_pushes diagram (_,Bishop) sq = bishop_attacks diagram sq
    | piece_pushes diagram (_,Queen) sq = queen_attacks diagram sq
    | piece_pushes _ (side,Pawn) sq = pawn_pushes side sq;

  fun piece_attacks _ (_,King) sq = king_attacks sq
    | piece_attacks _ (_,Knight) sq = knight_attacks sq
    | piece_attacks diagram (_,Rook) sq = rook_attacks diagram sq
    | piece_attacks diagram (_,Bishop) sq = bishop_attacks diagram sq
    | piece_attacks diagram (_,Queen) sq = queen_attacks diagram sq
    | piece_attacks _ (side,Pawn) sq = pawn_attacks side sq;
in
  fun possible_pushes diagram (side,piece) sq =
      let
(*Fen2ImgTrace3
        val () = trace "possible_pushes\n"
*)
        val Diagram on_square = diagram
        fun unoccupied sq' = not (Option.isSome (on_square sq'))
        val pushed_at = piece_pushes diagram (side,piece) sq
      in
        List.filter unoccupied pushed_at
      end;

  fun possible_captures diagram (side,piece) sq =
      let
(*Fen2ImgTrace3
        val () = trace "possible_captures\n"
*)
        val Diagram on_square = diagram
        fun enemy_occupies sq' =
            case on_square sq' of NONE => false | SOME (s,_) => s <> side
        val attacked = piece_attacks diagram (side,piece) sq
      in
        List.filter enemy_occupies attacked
      end;

  fun possible_moves diagram side_piece sq =
      possible_pushes diagram side_piece sq @
      possible_captures diagram side_piece sq;
end;

fun diagram_in_check diagram side =
    let
      val Diagram on_square = diagram
      fun attacks_our_king sq =
          case on_square sq of
            NONE => false
          | SOME (side',piece) =>
            side' <> side andalso
            List.exists
              (fn sq' => on_square sq' = SOME (side,King))
              (possible_captures diagram (side',piece) sq)
    in
      List.exists attacks_our_king all_squares
    end;

val piece_images =
    Lazy.memoize
    (fn () =>
     let
       fun load f =
           let
             val f = OS.Path.joinBaseExt {base = f, ext = SOME "ppm"}
             val f = OS.Path.joinDirFile {dir = !IMAGE_PATH, file = f}
           in
             Image.fromPlainPpm {filename = f}
           end

       val white_king = load "white_king"
       val white_queen = load "white_queen"
       val white_rook = load "white_rook"
       val white_bishop = load "white_bishop"
       val white_knight = load "white_knight"
       val white_pawn = load "white_pawn"
       val black_king = load "black_king"
       val black_queen = load "black_queen"
       val black_rook = load "black_rook"
       val black_bishop = load "black_bishop"
       val black_knight = load "black_knight"
       val black_pawn = load "black_pawn"
     in
       fn (Side.White,King) => white_king
        | (Side.White,Queen) => white_queen
        | (Side.White,Rook) => white_rook
        | (Side.White,Bishop) => white_bishop
        | (Side.White,Knight) => white_knight
        | (Side.White,Pawn) => white_pawn
        | (Side.Black,King) => black_king
        | (Side.Black,Queen) => black_queen
        | (Side.Black,Rook) => black_rook
        | (Side.Black,Bishop) => black_bishop
        | (Side.Black,Knight) => black_knight
        | (Side.Black,Pawn) => black_pawn
     end);

val empty_square =
    Image.Image
      {width = SQUARE_LENGTH,
       height = SQUARE_LENGTH,
       pixels = K Image.transparent};

val light_square = empty_square;

val dark_square =
    let
      fun f (Image.Coord {x,y}) _ =
          let
            val z = x + y - (SQUARE_LENGTH - 1)
            val z =
                if z mod STRIPE_PERIOD = STRIPE_PERIOD div 2 then 1.0 else 0.0
          in
            Image.Pixel {colour = STRIPE_COLOUR, opacity = z}
          end
    in
      Image.cmap f empty_square
    end;

fun piece_image p =
    let
      fun score (Image.Pixel {colour,opacity}) =
          opacity * (1.0 - Image.colourDiff colour Image.black)

      val image = piece_images () p
      val image = Image.greenScreen image
      val image = Image.centre {sub = empty_square, super = image}
      val shadow =
          Image.shadow
            {distance = BORDER_WIDTH,
             opacity = score,
             colour = Image.white}
          image
    in
      Image.centre {sub = Image.calculate shadow, super = image}
    end;

fun diagram_to_image (Diagram diagram) =
    let
      fun add_square dark ((rank,file),board) =
          if divides 2 (rank + file) <> dark then board
          else
            let
              val square = if dark then dark_square else light_square

              val square =
                  case diagram (Square {rank = rank, file = file}) of
                    NONE => square
                  | SOME p => Image.centre {sub = square, super = piece_image p}

              val x = (SQUARE_LENGTH - SQUARE_OVERLAP) * (file - 1)
              val y = (SQUARE_LENGTH - SQUARE_OVERLAP) * (RANKS - rank)
              val top_left = Image.Coord {x = x, y = y}
            in
              Image.superimpose
                {sub = board,
                 topLeft = top_left,
                 super = square}
            end

      val board =
          Image.Image
            {width = FILES * SQUARE_LENGTH - (FILES-1) * SQUARE_OVERLAP,
             height = RANKS * SQUARE_LENGTH - (RANKS-1) * SQUARE_OVERLAP,
             pixels = K (Image.solid Image.white)}

      val squares = cart (interval 1 RANKS) (interval 1 FILES)

      val board = List.foldl (add_square false) board squares

      val board = List.foldl (add_square true) board squares

      val board = Image.frame (Image.solid Image.white, FRAME_SPACE) board

      val board = Image.frame (Image.solid Image.black, FRAME_WIDTH) board
    in
      board
    end;

(* ------------------------------------------------------------------------- *)
(* Castling.                                                                 *)
(* ------------------------------------------------------------------------- *)

datatype castling_side = Kingside | Queenside

type castling_rights = castling_side -> bool

type castling = Side.side -> castling_rights

(* FEN order *)
val all_castling_sides = [Kingside,Queenside];

fun castling_side_to_char Kingside = #"K"
  | castling_side_to_char Queenside = #"Q";

val castling_rights_equal : castling_rights * castling_rights -> bool =
    fn (castling1,castling2) =>
       castling1 Kingside = castling2 Kingside andalso
       castling1 Queenside = castling2 Queenside;

val no_castling_rights : castling_rights = K false;

fun castling_rights_to_string castling =
    let
      val castling_sides = List.filter castling all_castling_sides
    in
      String.implode (List.map castling_side_to_char castling_sides)
    end;

val castling_equal : castling * castling -> bool =
    fn (castling1,castling2) =>
       castling_rights_equal (castling1 Side.White, castling2 Side.White)
       andalso
       castling_rights_equal (castling1 Side.Black, castling2 Side.Black);

val no_castling : castling = K no_castling_rights;

fun flip_castling castling : castling = castling o Side.opponent;

(* FEN syntax *)
fun castling_to_string castling =
    let
      val white = castling_rights_to_string (castling Side.White)
      val black = castling_rights_to_string (castling Side.Black)
      val black = String.map Char.toLower black
    in
      if white = "" andalso black = "" then "-" else white ^ black
    end;

local
  val white_pats =
      List.map (fn s => (castling_side_to_char s, s)) all_castling_sides;

  val black_pats = List.map (fn (c,s) => (Char.toLower c, s)) white_pats;

  fun mk_rights l s = mem s l;

  fun parse_rights l _ [] = (mk_rights l, [])
    | parse_rights l [] input = (mk_rights l, input)
    | parse_rights l ((pat,x) :: pats) (input as c :: rest) =
      if c = pat then parse_rights (x :: l) pats rest
      else parse_rights l pats input;
in
  fun string_to_castling "" = raise Error "string_to_castling"
    | string_to_castling "-" = no_castling
    | string_to_castling s =
      let
        val input = String.explode s
        val (white_rights,input) = parse_rights [] white_pats input
        val (black_rights,input) = parse_rights [] black_pats input
        val _ = List.null input orelse raise Error "string_to_castling"
      in
        fn Side.White => white_rights
         | Side.Black => black_rights
      end;
end;

(* ------------------------------------------------------------------------- *)
(* Positions.                                                                *)
(* ------------------------------------------------------------------------- *)

datatype position =
    Position of
      {diagram : diagram,
       to_move : Side.side,
       castling : castling,
       en_passant : square option,
       halfmove_clock : int,
       fullmove_number : int};

fun flip_en_passant NONE = NONE
  | flip_en_passant (SOME sq) = SOME (flip_square sq);

fun position_equal (position1,position2) =
    let
      val Position
            {diagram = diagram1,
             to_move = to_move1,
             castling = castling1,
             en_passant = en_passant1,
             halfmove_clock = halfmove_clock1,
             fullmove_number = fullmove_number1} = position1
      val Position
            {diagram = diagram2,
             to_move = to_move2,
             castling = castling2,
             en_passant = en_passant2,
             halfmove_clock = halfmove_clock2,
             fullmove_number = fullmove_number2} = position2
    in
      diagram_equal (diagram1,diagram2) andalso
      to_move1 = to_move2 andalso
      castling_equal (castling1,castling2) andalso
      en_passant1 = en_passant2 andalso
      halfmove_clock1 = halfmove_clock2 andalso
      fullmove_number1 = fullmove_number2
    end;

fun dest_position (Position position) = position;
val position_diagram = #diagram o dest_position
and position_to_move = #to_move o dest_position
and position_castling = #castling o dest_position
and position_en_passant = #en_passant o dest_position
and position_halfmove_clock = #halfmove_clock o dest_position
and position_fullmove_number = #fullmove_number o dest_position;

fun position_to_string position =
    let
      val Position
            {diagram, to_move, castling, en_passant,
             halfmove_clock, fullmove_number} = position
    in
      diagram_to_string diagram
      ^ Int.toString fullmove_number ^ ". "
      ^ Side.toLongString to_move ^ " "
      ^ castling_to_string castling ^ " "
      ^ (case en_passant of NONE => "-" | SOME sq => square_to_string sq) ^ " "
      ^ "+" ^ Int.toString halfmove_clock ^ "\n"
    end;

fun position_to_image (Position {diagram,...}) = diagram_to_image diagram;

fun flip_position position =
    let
      val Position
            {diagram, to_move, castling, en_passant,
             halfmove_clock, fullmove_number} = position
      val diagram = flip_diagram diagram
      val to_move = Side.opponent to_move
      val castling = flip_castling castling
      val en_passant = flip_en_passant en_passant
    in
      Position
        {diagram = diagram,
         to_move = to_move,
         castling = castling,
         en_passant = en_passant,
         halfmove_clock = halfmove_clock,
         fullmove_number = fullmove_number}
    end;

fun legal_position (Position {diagram,to_move,...}) =
    not (diagram_in_check diagram (Side.opponent to_move));

fun check_position (Position {diagram,to_move,...}) =
    diagram_in_check diagram to_move;

fun position_to_fen position =
    let
      val Position
            {diagram, to_move, castling, en_passant,
             halfmove_clock, fullmove_number} = position
    in
      diagram_to_fen_prefix diagram ^ " "
      ^ Side.toString to_move ^ " "
      ^ castling_to_string castling ^ " "
      ^ (case en_passant of NONE => "-" | SOME sq => square_to_string sq) ^ " "
      ^ Int.toString halfmove_clock ^ " "
      ^ Int.toString fullmove_number
    end;

val pp_position = Print.ppMap position_to_fen Print.ppString;

local
  open Parse;

  infixr 9 >>++
  infixr 8 ++
  infixr 7 >>
  infixr 6 ||

  val dash_parser = exactChar #"-";

  val space_parser = exactChar #" ";

  val digit_parser = some Char.isDigit >> (natFromString "digit_parser" o str);

  val number_parser =
      atLeastOne (some Char.isDigit) >>
      (natFromString "number_parser" o String.implode);

  val side_parser = maybe (total Side.fromChar);

  val side_piece_parser = maybe (total char_to_side_piece);

  val file_parser = maybe (total char_to_file);

  val rank_parser = maybe (total char_to_rank);

  val square_parser = (file_parser ++ rank_parser) >> mk_square;

  val castling_parser =
      many (some (not o Char.isSpace)) >> (string_to_castling o String.implode);

  val en_passant_parser = (dash_parser >> K NONE) || square_parser >> SOME;

  val halfmove_clock_parser = number_parser;

  val fullmove_number_parser = number_parser;
in
  val position_parser =
      (diagram_parser ++ space_parser
       ++ side_parser ++ space_parser
       ++ castling_parser ++ space_parser
       ++ en_passant_parser ++ space_parser
       ++ halfmove_clock_parser ++ space_parser
       ++ fullmove_number_parser ++ finished)
      >> (fn (d,(_,(m,(_,(c,(_,(e,(_,(h,(_,(n,_))))))))))) =>
          Position {diagram = d, to_move = m, castling = c, en_passant = e,
                    halfmove_clock = h, fullmove_number = n});
end;

val fen_to_position = fst o position_parser o input_chars;

val initial_fen_prefix = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR";

val initial_fen = initial_fen_prefix ^ " w KQkq - 0 1";

val initial_position = fen_to_position initial_fen;

(* ------------------------------------------------------------------------- *)
(* Moves                                                                     *)
(* ------------------------------------------------------------------------- *)

datatype move =
    Move of {piece : piece, from : square, to : square}
  | Promotion of {from : square, to : square, promotion : piece}
  | En_passant of {from : square, to : square}
  | Castling of castling_side;

fun initial_pawn_move (Move {piece = Pawn, from, to}) =
    Int.abs (square_rank from - square_rank to) = 2
  | initial_pawn_move _ = false;

fun apply_move move position =
    let
      val Position
            {diagram, to_move, castling, en_passant,
             halfmove_clock, fullmove_number} = position
      val Diagram on_square = diagram
      val to_move' = Side.opponent to_move
      val fullmove_number' =
          if to_move = Side.Black then fullmove_number + 1 else fullmove_number
    in
      case move of
        Move {piece,from,to} =>
        let
          fun on_square' sq =
              if sq = from then NONE
              else if sq = to then SOME (to_move,piece)
              else on_square sq
          val diagram' = Diagram on_square'
          fun castling' side =
              if side <> to_move then castling side
              else if piece = King then no_castling_rights
              else if piece = Rook then
                if from = king_rook_square to_move then
                  (fn Kingside => false | Queenside => castling side Queenside)
                else if from = queen_rook_square to_move then
                  (fn Kingside => castling side Kingside | Queenside => false)
                else castling side
              else castling side
          val en_passant' =
              if not (initial_pawn_move move) then NONE
              else
                let
                  val Square {file, rank = from_rank} = from
                  and Square {rank = to_rank, ...} = to
                  val rank = (from_rank + to_rank) div 2
                in
                  SOME (Square {file = file, rank = rank})
                end
          val is_capture = Option.isSome (on_square to)
          val halfmove_clock' =
              if piece = Pawn orelse is_capture then 0 else halfmove_clock + 1
        in
          Position
            {diagram = diagram',
             to_move = to_move',
             castling = castling',
             en_passant = en_passant',
             halfmove_clock = halfmove_clock',
             fullmove_number = fullmove_number'}
        end
      | Castling castling_side =>
        let
          val Square {file = king_file, rank = king_rank} = king_square to_move
          fun around_king i = mk_square (king_file + i, king_rank)
          fun shift_pieces i j k sq =
              if sq = around_king 0 then NONE
              else if sq = around_king i then SOME (to_move,King)
              else if sq = around_king j then NONE
              else if sq = around_king k then SOME (to_move,Rook)
              else on_square sq
          val on_square' =
              case castling_side of
                Kingside => shift_pieces 2 3 1
              | Queenside => shift_pieces ~2 ~4 ~1
          val diagram' = Diagram on_square'
          fun castling' side =
              if side = to_move then no_castling_rights else castling side
          val en_passant' = NONE
          val halfmove_clock' = halfmove_clock + 1
        in
          Position
            {diagram = diagram',
             to_move = to_move',
             castling = castling',
             en_passant = en_passant',
             halfmove_clock = halfmove_clock',
             fullmove_number = fullmove_number'}
        end
      | Promotion {from,to,promotion} =>
        let
          fun on_square' sq =
              if sq = from then NONE
              else if sq = to then SOME (to_move,promotion)
              else on_square sq
          val diagram' = Diagram on_square'
          val castling' = castling
          val en_passant' = NONE
          val halfmove_clock' = 0
        in
          Position
            {diagram = diagram',
             to_move = to_move',
             castling = castling',
             en_passant = en_passant',
             halfmove_clock = halfmove_clock',
             fullmove_number = fullmove_number'}
        end
      | En_passant {from,to} =>
        let
          val target_square =
              let
                val Square {rank,...} = from
                and Square {file,...} = to
              in
                Square {file = file, rank = rank}
              end
          fun on_square' sq =
              if sq = from then NONE
              else if sq = to then SOME (to_move,Pawn)
              else if sq = target_square then NONE
              else on_square sq
          val diagram' = Diagram on_square'
          val castling' = castling
          val en_passant' = NONE
          val halfmove_clock' = 0
        in
          Position
            {diagram = diagram',
             to_move = to_move',
             castling = castling',
             en_passant = en_passant',
             halfmove_clock = halfmove_clock',
             fullmove_number = fullmove_number'}
        end
    end;

fun all_legal_moves position =
    let
(*Fen2ImgTrace2
      val () = Print.trace pp_position "all_legal_moves: position" position
*)

      val Position {to_move,diagram,castling,en_passant,...} = position

      fun all_from (from,(side,piece),z) =
          if side <> to_move then z
          else
            List.map
              (fn to => Move {piece = piece, from = from, to = to})
              (possible_moves diagram (side,piece) from) @ z

      val promotion_rank = case to_move of Side.White => RANKS | Side.Black => 1

      fun promote (move as Move {piece = Pawn, from, to}) =
          if square_rank to = promotion_rank then
            List.map (fn p => Promotion {from = from, to = to, promotion = p})
            [Queen,Rook,Bishop,Knight]
          else [move]
        | promote move = [move]

      val castling_moves =
          let
            val Diagram on_square = diagram
            val Square {file=king_file, rank=king_rank} = king_square to_move
            fun around_king i = mk_square (king_file + i, king_rank)
            fun shift_king i sq =
                if sq = around_king i then SOME (to_move,King)
                else if sq = around_king 0 then NONE
                else on_square sq
            fun checked i = diagram_in_check (Diagram (shift_king i)) to_move
            val blocked = Option.isSome o on_square o around_king
            val castling_rights = castling to_move
            val moves = []
            val moves =
                if castling_rights Kingside andalso
                   List.all (not o checked) [0,1,2] andalso
                   List.all (not o blocked) [1,2]
                then Castling Kingside :: moves
                else moves
            val moves =
                if castling_rights Queenside andalso
                   List.all (not o checked) [~2,~1,0] andalso
                   List.all (not o blocked) [~3,~2,~1]
                then Castling Queenside :: moves
                else moves
          in
            moves
          end

      val initial_pawn_moves =
          let
            val pawn_rank = initial_pawn_rank to_move
            val pawn_vec = pawn_direction to_move
            val Diagram on_square = diagram
            fun around file i = mk_square (file, pawn_rank + i * pawn_vec)
            fun blocked file = Option.isSome o on_square o around file
            fun initial_pawn_move file =
                if on_square (around file 0) <> SOME (to_move,Pawn) then NONE
                else if List.exists (blocked file) [1,2] then NONE
                else SOME (Move {piece = Pawn, from = around file 0,
                                 to = around file 2})
          in
            List.mapPartial initial_pawn_move (interval 1 FILES)
          end

      val en_passant_moves =
          case en_passant of
            NONE => []
          | SOME to =>
            let
              val Diagram on_square = diagram
              fun is_pawn sq =
                  on_board sq andalso on_square sq = SOME (to_move,Pawn)
              val Square {file = to_file, rank = to_rank} = to
              val pawn_vec = pawn_direction to_move
              val from_rank = to_rank - pawn_vec
              val from_files = [to_file - 1, to_file + 1]
              val from_squares = List.map mk_square (cart from_files [from_rank])
              val from_squares = List.filter is_pawn from_squares
            in
              List.map (fn from => En_passant {from = from, to = to}) from_squares
            end

      val all_moves = fold_pieces all_from [] diagram
      val all_moves = List.concat (List.map promote all_moves)
      val all_moves = castling_moves @ all_moves
      val all_moves = initial_pawn_moves @ all_moves
      val all_moves = en_passant_moves @ all_moves
      val all_moves = List.map (fn m => (m, apply_move m position)) all_moves
    in
      List.filter (legal_position o snd) all_moves
    end;

fun checkmate_position position =
    check_position position andalso List.null (all_legal_moves position);

fun stalemate_position position =
    not (check_position position) andalso List.null (all_legal_moves position);

fun deduce_move position =
    let
(*Fen2ImgTrace2
      val () = trace "deduce_move\n"
*)
      val moves = all_legal_moves position
    in
      fn position' =>
         case List.find (fn (_,p) => position_equal (position',p)) moves of
           NONE => raise Bug "deduce_move: impossible move"
         | SOME (move,_) => move
    end;

local
  fun move_to_pure_san _ _ (Castling Kingside) = "O-O"
    | move_to_pure_san _ _ (Castling Queenside) = "O-O-O"
    | move_to_pure_san _ _ (Promotion {from,to,promotion}) =
      (if square_file from = square_file to then ""
       else file_to_string (square_file from) ^ "x") ^
      square_to_string to ^ "=" ^ piece_to_string promotion
    | move_to_pure_san _ _ (En_passant {from,to}) =
      file_to_string (square_file from) ^ "x" ^ square_to_string to
    | move_to_pure_san position moves (move as Move {piece,from,to}) =
      let
        val (position',moves) =
            case List.partition (equal move o fst) moves of
              ([(_,position')],moves) => (position', List.map fst moves)
            | _ => raise Bug "move_to_san: couldn't find move"
        val Position {diagram,to_move,...} = position
        val Diagram on_square = diagram
        val Position {diagram = diagram', to_move = to_move', ...} = position'
        val is_capture = Option.isSome (on_square to)
        val piece_sym = case piece of Pawn => "" | _ => piece_to_string piece
        val disambiguation_sym =
            if piece = Pawn then
              if is_capture then file_to_string (square_file from) else ""
            else
              let
                fun similar_moves (Move {piece = p, from = f, to = t}) =
                    if p = piece andalso t = to then SOME f else NONE
                  | similar_moves _ = NONE
                val similar = List.mapPartial similar_moves moves
                fun unambiguous pred = List.null (List.filter pred similar)
                val (file,rank) = dest_square from
              in
                if List.null similar then
                  ""
                else if unambiguous (equal file o square_file) then
                  file_to_string file
                else if unambiguous (equal rank o square_rank) then
                  rank_to_string rank
                else
                  square_to_string from
              end
        val capture_sym = if is_capture then "x" else ""
        val square_sym = square_to_string to
      in
        piece_sym
        ^ disambiguation_sym
        ^ capture_sym
        ^ square_sym
      end;
in
  fun move_to_san' position legal_moves move =
      let
        val position' = apply_move move position
        val status_symbol =
            if check_position position' then
              if checkmate_position position' then "#" else "+"
            else ""
      in
        move_to_pure_san position legal_moves move ^ status_symbol
      end;

  fun move_to_san position =
      move_to_san' position (all_legal_moves position);
end;

fun san_to_move position =
    let
      val moves = all_legal_moves position
      val moves = List.map (fn (m,_) => (move_to_san' position moves m, m)) moves
    in
      fn s =>
      case List.find (equal s o fst) moves of
        SOME (_,m) => m
      | NONE => raise Error "san_to_move"
    end;

(* ------------------------------------------------------------------------- *)
(* Games                                                                     *)
(* ------------------------------------------------------------------------- *)

type tag = {name : string, value : string}

type halfmove = {position : position, comment : string}

datatype result = White_wins | Black_wins | Draw

datatype game =
  Game of {tags : tag list, halfmoves : halfmove list, result : result option}

fun lookup_tag name (tags : tag list) =
    case List.find (equal name o #name) tags of
      NONE => NONE
    | SOME {value, ...} => SOME value;

fun tag_to_string {name,value} = "[" ^ name ^ " \"" ^ value ^ "\"]";

val pp_tag = Print.ppMap tag_to_string Print.ppString;

val pp_tags = Print.ppList pp_tag;

val all_results = [SOME White_wins, SOME Black_wins, SOME Draw, NONE];

fun result_to_string (SOME White_wins) = "1-0"
  | result_to_string (SOME Black_wins) = "0-1"
  | result_to_string (SOME Draw) = "1/2-1/2"
  | result_to_string NONE = "*";

fun string_to_result "1-0" = SOME White_wins
  | string_to_result "0-1" = SOME Black_wins
  | string_to_result "1/2-1/2" = SOME Draw
  | string_to_result "*" = NONE
  | string_to_result _ = raise Error "string_to_result";

fun game_result (Game {result,...}) = result;

fun game_length (Game {halfmoves,...}) = length halfmoves;

fun fens_to_game fens =
    let
      val tags = []
      val positions = List.map (fen_to_position o chomp) fens
      val halfmoves = List.map (fn p => {position = p, comment = ""}) positions
      val final_position = List.last positions
      val result =
          if checkmate_position final_position then
            SOME (if position_to_move final_position = Side.White
                  then Black_wins
                  else White_wins)
          else if stalemate_position final_position then SOME Draw
          else NONE
    in
      Game {tags = tags, halfmoves = halfmoves, result = result}
    end;

val pgn_date = Date.fmt "%Y.%m.%d";

val pgn_time = Date.fmt "%H:%M:%S";

fun pgn_move_number position =
    let
      val Position {to_move,fullmove_number,...} = position
      val dots = nChars #"." (if to_move = Side.White then 1 else 3)
    in
      Int.toString fullmove_number ^ dots
    end;

local
  fun pgn_comment "" = "" | pgn_comment s = "{" ^ s ^ "}";

  type lines = string list * string;

  val lines_empty : lines = ([],"");

  fun lines_add (lines,"") extra = (lines,extra)
    | lines_add (lines,line) extra =
      if size line + size extra >= 80 then (line :: lines, extra)
      else (lines, line ^ " " ^ extra);

  fun lines_final (lines,line) = List.rev (line :: lines);

  fun pgn_moves lines _ _ [] result =
      lines_final (lines_add lines (result_to_string result))
    | pgn_moves lines neednum position (halfmove :: halfmoves) result =
      let
        val {position = position', comment} = halfmove
        val move = move_to_san position (deduce_move position position')
        val extra =
            (if neednum then pgn_move_number position ^ " " else "") ^ move
            ^ (if comment = "" then "" else " " ^ pgn_comment comment)
        val lines = lines_add lines extra
        val neednum =
            comment <> "" orelse position_to_move position' = Side.White
      in
        pgn_moves lines neednum position' halfmoves result
      end;
in
  fun game_to_pgn game =
      let
        val Game {tags,halfmoves,result} = game
        val ({position,comment},halfmoves) = hdTl halfmoves
        val tags =
            if position_equal (position,initial_position) then tags
            else
              tags @
              [{name = "SetUp", value = "1"},
               {name = "FEN", value = position_to_fen position}]
        val tags_section = List.map tag_to_string tags
        val game_section =
            pgn_moves
              (lines_add lines_empty (pgn_comment comment))
              true position halfmoves result
      in
        String.concat
          (List.map (fn t => t ^ "\n") (tags_section @ [""] @ game_section))
      end;
end;

fun games_to_pgn filename =
    Stream.toTextFile filename o Stream.map game_to_pgn o Stream.fromList;

local
  open Parse;

  infixr 9 >>++
  infixr 8 ++
  infixr 7 >>
  infixr 6 ||

  fun pgn_escape "" = false
    | pgn_escape s =
      if String.sub (s,0) <> #"%" then false
      else (!PGN_ESCAPE (String.extract (s,1,NONE)); true);

  val input_chars =
      Stream.concat o Stream.map (Stream.fromList o String.explode) o
      Stream.filter (not o pgn_escape);

  fun padded_parser p = (manySpace ++ p ++ manySpace) >> (fn (_,(x,_)) => x);

  fun string_map_parser [] = error
    | string_map_parser ((s,x) :: rest) =
      (exactString s >> K x) || string_map_parser rest;

  val quoted_string_parser =
      let
        fun in_quote (true,l) =
            (exactChar #"\\" >> K (false, #"\\" :: l)
             || exactChar #"\"" >> K (false, #"\"" :: l)
             || any >> (fn c => raise Error ("bad escape sequence \\" ^ str c)))
          | in_quote (false,l) =
            (exactChar #"\\" >> K (true,l)
             || exactChar #"\n" >>
                (fn _ => raise Error "newline in quoted string")
             || some (not o equal #"\"") >> (fn c => (false, c :: l)))
      in
        (exactChar #"\"" ++ mmany in_quote (false,[]) ++ exactChar #"\"")
        >> (fn (_,((_,l),_)) => String.implode (List.rev l))
      end;

  val tag_name_parser =
      atLeastOne
        (some (fn #"_" => true | c => Char.isAlphaNum c)) >>
        String.implode;

  val tag_parser =
      (exactChar #"["
       ++ padded_parser tag_name_parser
       ++ padded_parser quoted_string_parser
       ++ exactChar #"]")
      >> (fn (_,(n,(v,_))) => {name = n, value = v});

  val comment_parser =
      let
        fun in_comment term l =
            some (fn c => c <> term)
            >> (fn c =>
                if not (Char.isSpace c) then c :: l
                else case l of #" " :: _ => l | _ => #" " :: l)

        fun cparser a b =
            (exactChar a ++ mmany (in_comment b) [] ++ exactChar b)
            >> (trim o String.implode o List.rev o fst o snd)
      in
        cparser #";" #"\n"
        || cparser #"{" #"}"
      end;

  val move_number_parser = exactString o pgn_move_number

  fun mk_move_parser position =
      let
        val moves = all_legal_moves position
        val moves = List.map (fn (m,p) => (move_to_san' position moves m, p)) moves
(*Fen2ImgTrace2
        val ppMoves = Print.ppMap (List.map fst) (Print.ppList Print.ppString)
        val () = Print.trace ppMoves "mk_move_parser: moves" moves
*)
      in
        string_map_parser moves
      end;

  val result_parser =
      string_map_parser (List.map (fn r => (result_to_string r, r)) all_results);

  fun initial_game (tags : tag list) =
      let
(*Fen2ImgTrace3
        val () = Print.trace pp_tags "initial_game: tags" tags
*)
        val (setup,tags) = List.partition (equal "SetUp" o #name) tags
        val (fen,tags) = List.partition (equal "FEN" o #name) tags
        val halfmoves = []
        val position =
            case setup of
              [] => initial_position
            | [{value = "0", ...}] => initial_position
            | [{value = "1", ...}] =>
              (case fen of
                 [{value = f, ...}] => fen_to_position f
               | _ => raise Error "tag [SetUp \"1\"], but no FEN tag")
            | {value,...} :: _ =>
              raise Error ("bad value for SetUp tag: \"" ^ value ^ "\"")
        val halfmove = {position = position, comment = ""}
        val neednum = true
        val move_parser = mk_move_parser position
        val seenresult = false
      in
        (tags,halfmoves,halfmove,neednum,move_parser,seenresult)
      end;

  fun track_game state =
      let
        val (tags,halfmoves,halfmove,neednum,move_parser,seenresult) = state
        val {position,comment} = halfmove
        val game_comment_parser = padded_parser comment_parser
        val game_move_parser =
            if seenresult then error
            else if not neednum then padded_parser move_parser
            else (padded_parser (move_number_parser position)
                  ++ padded_parser move_parser) >> snd
        val game_result_parser =
            if seenresult then error else padded_parser result_parser
      in
        game_comment_parser
        >> (fn comment' =>
            let
              val comment = (if comment = "" then "" else " ") ^ comment'
              val halfmove = {position = position, comment = comment}
            in
              (tags,halfmoves,halfmove,true,move_parser,seenresult)
            end)
        || game_move_parser
           >> (fn position' =>
               let
                 val halfmoves = halfmove :: halfmoves
                 val halfmove = {position = position', comment = ""}
                 val neednum = position_to_move position' = Side.White
                 val move_parser = mk_move_parser position'
               in
                 (tags,halfmoves,halfmove,neednum,move_parser,seenresult)
               end)
        || game_result_parser
           >> (fn result =>
               let
                 val tags =
                     case lookup_tag "Result" tags of
                       SOME result_tag =>
                       if result_to_string result = result_tag then tags
                       else raise Error "Result tag doesn't match game result"
                     | NONE =>
                       {name = "Result", value = result_to_string result}
                       :: tags
               in
                 (tags,halfmoves,halfmove,neednum,move_parser,true)
               end)
      end;

  fun finalize_game (_,_,_,_,_,false) = raise NoParse
    | finalize_game (tags,halfmoves,halfmove,_,_,true) =
      let
        val result =
            case List.filter (equal "Result" o #name) tags of
              [{value = result, ...}] => string_to_result result
            | [] => raise Bug "game_parser: no Result tag"
            | _ => raise Error "multiple Result tags"
        val halfmoves = List.rev (halfmove :: halfmoves)
      in
        Game {tags = tags, halfmoves = halfmoves, result = result}
      end;

  val game_parser =
      ((many (padded_parser tag_parser) >> initial_game)
       >>++ mmany track_game) >> finalize_game;

  val games_parser = (many game_parser ++ finished) >> fst;

  val pgn_stream_to_games = fst o games_parser o input_chars;
in
  fun pgn_to_game s =
      case pgn_stream_to_games (Stream.singleton s) of
        [] => raise Error "pgn_to_game: no games"
      | [game] => game
      | _ => raise Error "pgn_to_game: multiple games";

  fun pgn_to_games n = pgn_stream_to_games (Stream.fromTextFile n);
end;

local
  fun mk_dir path = OS.FileSys.mkDir path handle OS.SysErr _ => ();

  fun take_row [] = raise Bug "game_to_html.take_two"
    | take_row [x] = ((SOME x, NONE), [])
    | take_row (x :: (xs as y :: ys)) =
      if position_to_move (#position x) = Side.White then ((SOME x, NONE), xs)
      else ((SOME x, SOME y), ys);

  fun halfmove_to_html _ _ _ NONE = []
    | halfmove_to_html (path,basename) prev n (SOME {position,comment}) =
      let
        val Position {to_move,...} = position
        val name =
            Int.toString (if to_move = Side.White then 2 * n - 1 else 2 * n)
        val image = position_to_image position
        val Image.Image {width,...} = image

        val () =
            let
              val d = OS.Path.concat (path,basename)
              val f = OS.Path.joinBaseExt {base = name, ext = SOME "ppm"}
              val f = OS.Path.joinDirFile {dir = d, file = f}
            in
              Image.toPlainPpm
                {filename = f, background = Image.white, image = image}
            end

        val attrs =
            Html.fromListAttrs
              [("align","left"),
               ("width", Int.toString width)]

        val img =
            let
              val f = OS.Path.joinBaseExt {base = name, ext = SOME "png"}
              val f = OS.Path.joinDirFile {dir = basename, file = f}
              val a = Html.fromListAttrs [("src",f),("alt","")]
            in
              Html.Img a
            end

        val san =
            case prev of
              NONE =>
              [Html.Strong [Html.Text (Side.toLongString to_move ^ " to move")]]
            | SOME {position = prev_position, ...} =>
              let
                val move = deduce_move prev_position position
                val move = move_to_san prev_position move
                val number = pgn_move_number prev_position
              in
                [Html.Strong [Html.Text (number ^ " " ^ move)]]
              end

        val comm = if comment = "" then [] else [Html.Small [Html.Text comment]]
      in
        [Html.Table
           (Html.tightTableAttrs,
            [Html.TableRow [Html.TableEntry (attrs, Html.Inline san)],
             Html.TableRow
               [Html.TableEntry
                  (Html.singletonAttrs ("height","10"), Html.emptyFlow)],
             Html.TableRow [Html.TableEntry (attrs, Html.Block [img])],
             Html.TableRow
               [Html.TableEntry
                  (Html.singletonAttrs ("height","10"), Html.emptyFlow)],
             Html.TableRow [Html.TableEntry (attrs, Html.Inline comm)]])]
      end;

  val pad_row =
      Html.TableRow
        [Html.TableEntry
           (Html.singletonAttrs ("height","50"), Html.emptyFlow)];

  fun halfmoves_to_html _ rows _ _ [] = List.rev rows
    | halfmoves_to_html path rows prev n halfmoves =
      let
        fun entrify x =
            Html.TableEntry (Html.singletonAttrs ("valign","top"), Html.Block x)

        val ((b,w),halfmoves) = take_row halfmoves

        val b_entry = entrify (halfmove_to_html path prev n b)
        val w_entry = entrify (halfmove_to_html path b (n + 1) w)

        val row =
            Html.TableRow
              [b_entry,
               Html.TableEntry
                 (Html.singletonAttrs ("width","50"), Html.emptyFlow),
               w_entry]

        val (prev,n) = case w of SOME _ => (w, n + 1) | NONE => (b,n)

        val rows = row :: (if List.null rows then rows else pad_row :: rows)
      in
        halfmoves_to_html path rows prev n halfmoves
      end;

  fun result_to_html result =
      let
        val attrs = Html.fromListAttrs [("colspan","3"),("align","center")]
        val entry = Html.Big [Html.Strong [Html.Text (result_to_string result)]]
      in
        Html.TableRow [Html.TableEntry (attrs, Html.Inline [entry])]
      end;
in
  fun game_to_html {path,basename} game =
      let
        val () = mk_dir (OS.Path.concat (path,basename))
        val Game {tags,halfmoves,result} = game
        val rows = halfmoves_to_html (path,basename) [] NONE 1 halfmoves
        val rows = rows @ [result_to_html result]
      in
        Html.Table (Html.tightTableAttrs,rows)
      end;
end;

local
  val TITLE : string option ref = ref NONE
  and HEADING : string option ref = ref NONE
  and SUBHEADING : string option ref = ref NONE
  and FOOTER : string option ref = ref NONE;

  fun html_escape escape =
      let
        val escape = chomp escape

        fun update_cell cell prefix =
            case total (destPrefix (prefix ^ " ")) escape of
              SOME x => cell := SOME x
            | NONE => ()

        val () = update_cell TITLE "TITLE"
        val () = update_cell HEADING "HEADING"
        val () = update_cell SUBHEADING "SUBHEADING"
        val () = update_cell FOOTER "FOOTER"
      in
        ()
      end;

  fun read_pgn (path,basename) =
      let
        val f = OS.Path.joinBaseExt {base = basename, ext = SOME "pgn"}
        val f = OS.Path.joinDirFile {dir = path, file = f}
      in
        withRef (PGN_ESCAPE,html_escape) pgn_to_games {filename = f}
      end;

  fun mk_title tags =
      case !TITLE of
        SOME title => title
      | NONE =>
        case (lookup_tag "White" tags, lookup_tag "Black" tags) of
          (SOME w, SOME b) => w ^ " vs " ^ b
        | _ => "Chess game";

  fun mk_heading tags =
      let
        val heading =
            case !HEADING of
              SOME heading => heading
            | NONE => mk_title tags
      in
        [Html.H1 [Html.Text heading]]
      end;

  fun mk_subheading tags =
      case !SUBHEADING of
        SOME subheading => [Html.H2 [Html.Text subheading]]
      | NONE => [];

  fun mk_pgn_link basename =
      let
        val f = OS.Path.joinBaseExt {base = basename, ext = SOME "pgn"}
      in
        Html.Para
          (Html.emptyAttrs,
           [Html.Text "Also available in ",
            Html.Anchor (Html.hrefAttrs f, [Html.Text "PGN notation"]),
            Html.Text "."])
      end;

  fun mk_footer tags =
      let
        val footer =
            case !FOOTER of
              SOME footer => footer
            | NONE =>
              let
                val now = Date.fromTimeLocal (Time.now ())
                val today = Date.fmt "%d %B %Y" now
                val today =
                    if isPrefix "0" today then String.extract (today,1,NONE)
                    else today
                val gilith = "<a href=\"http://www.gilith.com\">Gilith</a>"
                val advert = "generated by " ^ gilith ^ " on " ^ today
              in
                case lookup_tag "Annotator" tags of
                  SOME annotator =>
                  "Annotations by " ^ annotator ^ ", page " ^ advert ^ "."
                | NONE => "Page " ^ advert ^ "."
              end;
      in
        [Html.Hr, Html.Para (Html.emptyAttrs, [Html.Em [Html.Raw footer]])]
      end;
in
  fun pgn_to_html {path,basename} =
      case read_pgn (path,basename) of
        [] => raise Error "pgn_to_html: no games!"
      | [game] =>
        let
          val Game {tags,halfmoves,result} = game
          val title = mk_title tags
          val heading = mk_heading tags
          val subheading = mk_subheading tags
          val pgn_link = mk_pgn_link basename
          val table = game_to_html {path = path, basename = basename} game
          val footer = mk_footer tags
          val html_doc =
              Html.Html
                (Html.Head (Html.Title title, [], []),
                 Html.Body
                   (Html.emptyAttrs,
                    heading
                    @ subheading
                    @ [pgn_link]
                    @ [table]
                    @ footer))

          val html_name =
              OS.Path.joinBaseExt {base = basename, ext = SOME "html"}
          val html_name = OS.Path.joinDirFile {dir = path, file = html_name}

          val html_strm = Print.toStream Html.pp html_doc
        in
          Stream.toTextFile {filename = html_name} html_strm
        end
      | _ => raise Error "pgn_to_html: multiple games";
end;

end
