(* ========================================================================= *)
(* IMAGES                                                                    *)
(* Copyright (c) 2004 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature Image =
sig

(* ------------------------------------------------------------------------- *)
(* Image types.                                                              *)
(* ------------------------------------------------------------------------- *)

datatype colour = Colour of {red : real, green : real, blue : real}

datatype pixel = Pixel of {colour : colour, opacity : real}

datatype coord = Coord of {x : int, y : int}  (* Top left is (0,0) *)

datatype image = Image of {width : int, height : int, pixels : coord -> pixel}

(* ------------------------------------------------------------------------- *)
(* Colours.                                                                  *)
(* ------------------------------------------------------------------------- *)

val grey : real -> colour

val black : colour

val white : colour

val red : colour

val green : colour

val blue : colour

(* Uses sum of squares on RGB differences, normalized to be between 0 and 1 *)
val colourDiff : colour -> colour -> real

val colourMult : colour -> colour -> colour

val fromRgbColour : int * int * int -> colour

(* ------------------------------------------------------------------------- *)
(* Pixels.                                                                   *)
(* ------------------------------------------------------------------------- *)

val transparent : pixel

val solid : colour -> pixel

val pixelMap : (colour -> colour) -> pixel -> pixel

val superimposePixels : pixel -> pixel -> pixel

(* ------------------------------------------------------------------------- *)
(* Image transformations.                                                    *)
(* ------------------------------------------------------------------------- *)

val calculate : image -> image

val cmap : (coord -> pixel -> pixel) -> image -> image

val map : (pixel -> pixel) -> image -> image

val crop : {topLeft : coord, bottomRight : coord} -> image -> image

val cropDiagonal : image -> image

val chopVertical : {top : int, size : int} -> image -> image

val chopHorizontal : {left : int, size : int} -> image -> image

val superimpose : {sub : image, topLeft : coord, super : image} -> image

val centre : {sub : image, super : image} -> image

val frame : pixel * int -> image -> image

val colourScreen : colour * real -> image -> image

val greenScreen : image -> image

val shadow :
    {distance : real, opacity : pixel -> real, colour : colour} ->
    image -> image

val mirrorHorizontal : image -> image

val mirrorVertical : image -> image

val mirrorDiagonal : image -> image

(* ------------------------------------------------------------------------- *)
(* I/O.                                                                      *)
(* ------------------------------------------------------------------------- *)

val fromPlainPpm : {filename : string} -> image

val toPlainPpm : {filename : string, background : colour, image : image} -> unit

end
