(* ========================================================================= *)
(* THE PLAYERS OF A TWO PLAYER GAME                                          *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Side :> Side =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Sides.                                                                    *)
(* ------------------------------------------------------------------------- *)

datatype side = Black | White;

fun compare (Black,Black) = EQUAL
  | compare (Black,White) = LESS
  | compare (White,Black) = GREATER
  | compare (White,White) = EQUAL;

fun equal Black Black = true
  | equal Black White = false
  | equal White Black = false
  | equal White White = true;

val allSides = [Black,White];

(*GomiDebug
val () = if sort compare allSides = allSides then ()
         else raise Bug "Side.allSides: out of order";
*)

fun isBlack Black = true
  | isBlack White = false;

fun isWhite Black = false
  | isWhite White = true;

fun opponent Black = White
  | opponent White = Black;

(* ------------------------------------------------------------------------- *)
(* Generating random values.                                                 *)
(* ------------------------------------------------------------------------- *)

fun random () = if Portable.randomBool () then Black else White;

(* ------------------------------------------------------------------------- *)
(* Parsing and pretty-printing.                                              *)
(* ------------------------------------------------------------------------- *)

fun toChar Black = #"B"
  | toChar White = #"W";

fun fromChar c =
    case Char.toUpper c of
      #"B" => Black
    | #"W" => White
    | _ => raise Error "Side.fromChar";

fun toLongString Black = "Black"
  | toLongString White = "White";

fun fromLongString s =
    case String.map Char.toUpper s of
      "BLACK" => Black
    | "WHITE" => White
    | _ => raise Error "Side.fromLongString";

fun toString s = str (toChar s);

fun fromString s =
    case total fromLongString s of
      SOME s => s
    | NONE =>
      case (case String.explode s of [c] => total fromChar c | _ => NONE) of
        SOME s => s
      | NONE => raise Error "Side.fromString";

val pp = Print.ppMap toChar Print.ppChar;

(* ------------------------------------------------------------------------- *)
(* Storing data for both sides.                                              *)
(* ------------------------------------------------------------------------- *)

type 'a sides = {black : 'a, white : 'a};

fun sidesCompare cmp ({black = b1, white = w1}, {black = b2, white = w2}) =
    case cmp (b1,b2) of
      LESS => LESS
    | EQUAL => cmp (w1,w2)
    | GREATER => GREATER;

fun sidesEqual eq {black = b1, white = w1} {black = b2, white = w2} =
    eq b1 b2 andalso eq w1 w2;

fun pickSides {black,white} = fn Black => black | White => white;

fun mapSides f {black,white} = {black = f black, white = f white};

fun appSides f {black,white} = (f black; f white);

fun switchSides {black,white} = {black = white, white = black};

fun normalizeSides Black sides = sides
  | normalizeSides White sides = switchSides sides;

end
