(* ========================================================================= *)
(* FEN2IMG: CREATING A CHESS DIAGRAM FROM FEN NOTATION                       *)
(* Copyright (c) 2001 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

open Useful;

(* ------------------------------------------------------------------------- *)
(* The program name.                                                         *)
(* ------------------------------------------------------------------------- *)

val PROGRAM = "fen2img";

(* ------------------------------------------------------------------------- *)
(* Program options.                                                          *)
(* ------------------------------------------------------------------------- *)

datatype input =
    FileInput of {filename : string}
  | StringInput of string;

val INPUT = ref (FileInput {filename = "-"});

val OUTPUT = ref "-";

local
  open Useful Options;
in
  val specialOptions =
    [{switches = ["-i","--image-path"], arguments = ["PATH"],
      description = "the location of the chess piece images",
      processor =
        beginOpt (stringOpt endOpt) (fn _ => fn s => Chess.IMAGE_PATH := s)},
     {switches = ["-s","--input-string"], arguments = ["FEN"],
      description = "the input position in FEN notation",
      processor =
        beginOpt (stringOpt endOpt) (fn _ => fn s => INPUT := StringInput s)},
     {switches = ["-f","--input-file"], arguments = ["FILE"],
      description = "the file containing the input position",
      processor =
        beginOpt (stringOpt endOpt)
          (fn _ => fn s => INPUT := FileInput {filename = s})},
     {switches = ["-o","--output"], arguments = ["FILE"],
      description = "the output file in PPM image format",
      processor =
        beginOpt (stringOpt endOpt) (fn _ => fn s => OUTPUT := s)}];
end;

val VERSION = "1.1";

val versionString = "fen2img "^VERSION^" (release 20180810)"^"\n";

val programOptions =
    {name = PROGRAM,
     version = versionString,
     header = "usage: "^PROGRAM^" [option ...]\n",
     footer = "Creates a chess diagram from a position written in " ^
              "FEN notation.\n",
     options = specialOptions @ Options.basicOptions};

fun exit x : unit = Options.exit programOptions x;
fun succeed () = Options.succeed programOptions;
fun fail mesg = Options.fail programOptions mesg;
fun usage mesg = Options.usage programOptions mesg;

val (opts,work) =
    Options.processOptions programOptions (CommandLine.arguments ());

val () = if List.null work then () else usage "too many arguments";

(* ------------------------------------------------------------------------- *)
(* Top level.                                                                *)
(* ------------------------------------------------------------------------- *)

val () =
let
  val fen =
      case !INPUT of
        StringInput s => s
      | FileInput f =>
        chomp (String.concat (Stream.toList (Stream.fromTextFile f)))

  val diagram = Chess.fen_prefix_to_diagram fen

  val image = Chess.diagram_to_image diagram

  val () =
      Image.toPlainPpm
        {filename = !OUTPUT,
         image = image,
         background = Image.white}
in
  exit {message = NONE, usage = false, success = true}
end
handle Error s => die (PROGRAM^" failed:\n" ^ s)
     | Bug s => die ("BUG found in "^PROGRAM^" program:\n" ^ s);
