(* ========================================================================= *)
(* CREATING A WEB PAGE ILLUSTRATING A CHESS GAME IN PGN NOTATION             *)
(* Copyright (c) 2004 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

(* ------------------------------------------------------------------------- *)
(* Structures                                                                *)
(* ------------------------------------------------------------------------- *)

open Useful;

infixr ## |->

(* ------------------------------------------------------------------------- *)
(* The program name                                                          *)
(* ------------------------------------------------------------------------- *)

val PROGRAM = "pgn2html";

(* ------------------------------------------------------------------------- *)
(* The trace system                                                          *)
(* ------------------------------------------------------------------------- *)

val () = trace_level := 0;
fun chatting l = tracing {module = PROGRAM, level = l};
fun chat s = (trace s; true);

(* ------------------------------------------------------------------------- *)
(* Helper functions                                                          *)
(* ------------------------------------------------------------------------- *)


(* ------------------------------------------------------------------------- *)
(* Program options                                                           *)
(* ------------------------------------------------------------------------- *)

local
  open Useful Options;
in
  val special_options : opt list =
      [{switches = ["-i","--image-path"], arguments = ["PATH"],
        description = "the location of the chess piece images",
        processor = begin_proc (string_proc end_proc)
        (fn _ => fn h => Chess.IMAGE_PATH := h)}];
end;

val version_string = PROGRAM^" v1.0\n";

val program_options =
    {name    = PROGRAM,
     version = version_string,
     header  = "usage: "^PROGRAM^" [-i] game.pgn...\n\n",
     footer  = "\nCreates a web page illustrating a chess game " ^
               "written in PGN notation.\n",
     options = special_options @ Options.basic_options};

fun succeed () = Options.succeed program_options;
fun version () = Options.version program_options;
fun fail mesg = Options.fail program_options mesg;
fun usage mesg = Options.usage program_options mesg;

val (opts,work) =
  Options.process_options program_options (CommandLine.arguments ());

val () = if null work then version () else ();

(* ------------------------------------------------------------------------- *)
(* Top level                                                                 *)
(* ------------------------------------------------------------------------- *)

fun pgn2html pgn =
    let
      val {dir = path, file = filename} = OS.Path.splitDirFile pgn
      val {base = basename, ext} = OS.Path.splitBaseExt filename
      val _ = ext = SOME "pgn" orelse
              raise Error ("input file "^pgn^" does not have .pgn extension")
    in
      Chess.pgn_to_html {path = path, basename = basename}
    end;

val () =
let
  val () = app pgn2html work
in
  succeed ()
end
handle Error s => die (PROGRAM^" failed:\n" ^ s)
     | Bug s => die ("BUG found in "^PROGRAM^" program:\n" ^ s);
