(* ========================================================================= *)
(* GO BOARDS                                                                 *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Board :> Board =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Blocks of connected stones.                                               *)
(* ------------------------------------------------------------------------- *)

datatype block =
    Block of
      {stones : PointSet.set,
       neighbours : PointSet.set,
       liberties : PointSet.set};

type singletonBlock =
     {stone : Point.point,
      neighbours : PointSet.set,
      liberties : PointSet.set};

fun blockStones (Block {stones = s, ...}) = s;

fun blockNeighbours (Block {neighbours = n, ...}) = n;

fun liberties (Block {liberties = l, ...}) = l;

fun countLiberties block = PointSet.size (liberties block);

fun atari block = countLiberties block = 1;

fun decreaseLiberties (Block {stones,neighbours,liberties}) point =
    Block
      {stones = stones,
       neighbours = neighbours,
       liberties = PointSet.delete liberties point};

fun increaseLiberties (Block {stones,neighbours,liberties}) points =
    let
      val newLiberties = PointSet.intersect neighbours points
    in
      Block
        {stones = stones,
         neighbours = neighbours,
         liberties = PointSet.union liberties newLiberties}
    end;

local
  fun union (Block {stones,neighbours,liberties}, (s,n,l)) =
      (PointSet.union stones s, PointSet.union neighbours n,
       PointSet.union liberties l);
in
  fun mergeBlocks [] {stone = s, neighbours = n, liberties = l} =
      Block {stones = PointSet.singleton s, neighbours = n, liberties = l}
    | mergeBlocks blocks {stone,neighbours,liberties} =
      let
        val s_n_l = (PointSet.singleton stone, neighbours, liberties)

        val (s,n,l) = List.foldl union s_n_l blocks

        val n = PointSet.delete n stone
        and l = PointSet.delete l stone
      in
        Block {stones = s, neighbours = n, liberties = l}
      end;
end;

(* ------------------------------------------------------------------------- *)
(* A type of go boards.                                                      *)
(* ------------------------------------------------------------------------- *)

datatype board =
    Board of
      {fixed :
         {dimensions : Dimensions.dimensions,
          points : PointSet.set,
          neighbours : PointSet.set PointMap.map},
       emptyPoints : PointSet.set,
       stones : Point.point PointMap.map Side.sides,
       blocks : block PointMap.map Side.sides};

fun empty dim =
    let
      val points = Dimensions.points dim

      val fixed =
          {dimensions = dim,
           points = points,
           neighbours = Dimensions.map (Dimensions.neighbours dim) dim}
    in
      Board
        {fixed = fixed,
         emptyPoints = points,
         stones = {black = PointMap.new (), white = PointMap.new ()},
         blocks = {black = PointMap.new (), white = PointMap.new ()}}
    end;

fun compare (Board {stones = s1, ...}, Board {stones = s2, ...}) =
    let
      val n1 = Side.mapSides PointMap.size s1
      and n2 = Side.mapSides PointMap.size s2
    in
      case Side.sidesCompare Int.compare (n1,n2) of
        LESS => LESS
      | EQUAL => Side.sidesCompare (PointMap.compare (K EQUAL)) (s1,s2)
      | GREATER => GREATER
    end;

fun equal board1 board2 = compare (board1,board2) = EQUAL;

fun switchSides (Board {fixed,emptyPoints,stones,blocks}) =
    Board
      {fixed = fixed,
       emptyPoints = emptyPoints,
       stones = Side.switchSides stones,
       blocks = Side.switchSides blocks};

(* ------------------------------------------------------------------------- *)
(* Geometry.                                                                 *)
(* ------------------------------------------------------------------------- *)

fun dimensions (Board {fixed = {dimensions = d,...}, ...}) = d;

fun points (Board {fixed = {points = s,...}, ...}) = s;

fun neighbours (Board {fixed = {neighbours = m,...}, ...}) p =
    PointMap.get m p;

(* ------------------------------------------------------------------------- *)
(* Examining the stones on the board.                                        *)
(* ------------------------------------------------------------------------- *)

fun emptyPoints (Board {emptyPoints = e, ...}) = e;

fun peekBlock (Board {stones,...}) p =
    let
      val {black,white} = stones
    in
      case PointMap.peek black p of
        SOME i => SOME (Side.Black,i)
      | NONE =>
        case PointMap.peek white p of
          SOME i => SOME (Side.White,i)
        | NONE => NONE
    end;

fun getBlock (Board {blocks,...}) side i =
    case PointMap.peek (Side.pickSides blocks side) i of
      SOME block => block
    | NONE => raise Bug "Board.getBlock: missing";

fun peek board p =
    case peekBlock board p of
      NONE => NONE
    | SOME (side,i) => SOME (side, getBlock board side i);

fun destStone board p =
    case peekBlock board p of
      SOME (side,_) => SOME side
    | NONE => NONE;

fun isStone board p = Option.isSome (destStone board p);

fun isEmpty board p = not (Option.isSome (peekBlock board p));

fun countStones (Board {stones,...}) = Side.mapSides PointMap.size stones;

fun countBlocks (Board {blocks,...}) = Side.mapSides PointMap.size blocks;

fun foldStones f b (Board {stones = {black,...}, ...}) Side.Black =
    PointMap.foldl (fn (p,_,z) => f (p,z)) b black
  | foldStones f b (Board {stones = {white,...}, ...}) Side.White =
    PointMap.foldl (fn (p,_,z) => f (p,z)) b white;

fun foldBlocks f b (Board {blocks = {black,...}, ...}) Side.Black =
    PointMap.foldl (fn (_,x,z) => f (x,z)) b black
  | foldBlocks f b (Board {blocks = {white,...}, ...}) Side.White =
    PointMap.foldl (fn (_,x,z) => f (x,z)) b white;

fun status board point =
    case destStone board point of
      SOME side => Status.Stone side
    | NONE =>
      let
        val points = neighbours board point
        val (point,points) = PointSet.deletePick points
      in
        case destStone board point of
          NONE => Status.Seki
        | SOME side =>
          let
            fun agrees point =
                case destStone board point of
                  NONE => false
                | SOME side' => Side.equal side side'
          in
            if PointSet.all agrees points then Status.Eye side
            else Status.Seki
          end
      end;

(* ------------------------------------------------------------------------- *)
(* Altering the stones on the board.                                         *)
(* ------------------------------------------------------------------------- *)

local
  fun classifyNeighbours {permitSuicide} board point =
      let
        val _ = isEmpty board point orelse raise Error "point not empty"

        fun insert (i_b as (i,_)) l =
            if List.exists (Point.equal i o fst) l then l else i_b :: l

        fun classify (p,(l,r,f,c,e)) =
            case peekBlock board p of
              SOME (side,i) =>
              let
                val b = getBlock board side i
              in
                case side of
                  Side.Black => (l, r, insert (i,b) f, c, e)
                | Side.White =>
                  let
                    val r = PointSet.add r p
                  in
                    if atari b then (l, r, f, insert (i,b) c, e)
                    else (l, r, f, c, insert (i,b) e)
                  end
              end
            | NONE => (PointSet.add l p, r, f, c, e)

        val neighbour_points = neighbours board point

        val (empty_points,enemy_points,
             friendly_blocks,captured_blocks,enemy_blocks) =
            PointSet.foldl
              classify (PointSet.empty,PointSet.empty,[],[],[])
              neighbour_points

        val suicide =
            PointSet.null empty_points andalso
            List.null captured_blocks andalso
            List.all (atari o snd) friendly_blocks andalso
            (permitSuicide orelse raise Error "suicide move")
      in
        {suicide = suicide,
         empty_points = empty_points,
         enemy_points = enemy_points,
         friendly_blocks = friendly_blocks,
         captured_blocks = captured_blocks,
         enemy_blocks = enemy_blocks}
      end;

  fun placeBlack permitSuicide board point =
      let
        fun decrease_liberties ((i,b),m) =
            PointMap.insert m (i, decreaseLiberties b point)

        fun remove_blocks ((i,b),(m,s)) =
            (PointMap.delete m i, PointSet.union s (blockStones b))

        fun remove_stones (p,m) = PointMap.delete m p

        val {suicide,empty_points,enemy_points,
             friendly_blocks,captured_blocks,enemy_blocks} =
            classifyNeighbours permitSuicide board point

        val Board {fixed,emptyPoints,stones,blocks} = board

        val {black = black_stones, white = white_stones} = stones
        and {black = black_blocks, white = white_blocks} = blocks
      in
        if suicide then
          case friendly_blocks of
            [] => board
          | suicide_blocks =>
            let
              val (black_blocks,suicide_stones) =
                  List.foldl
                    remove_blocks (black_blocks,PointSet.empty) suicide_blocks

              val emptyPoints = PointSet.union emptyPoints suicide_stones

              val black_stones =
                  PointSet.foldl remove_stones black_stones suicide_stones

              val white_blocks =
                  PointMap.transform
                    (fn b => increaseLiberties b suicide_stones) white_blocks
            in
              Board
                {fixed = fixed,
                 emptyPoints = emptyPoints,
                 stones = {black = black_stones, white = white_stones},
                 blocks = {black = black_blocks, white = white_blocks}}
            end
        else
          let
            val white_blocks =
                List.foldl decrease_liberties white_blocks enemy_blocks

            val (white_blocks,captured_stones) =
                List.foldl
                  remove_blocks (white_blocks,PointSet.empty) captured_blocks

            val emptyPoints =
                PointSet.union
                  (PointSet.delete emptyPoints point) captured_stones

            val white_stones =
                PointSet.foldl remove_stones white_stones captured_stones

            val block =
                mergeBlocks
                  (List.map snd friendly_blocks)
                  {stone = point,
                   neighbours = PointSet.union empty_points enemy_points,
                   liberties = empty_points}

            val (black_stones,block_id) =
                let
                  fun find_id ((id, Block {stones,...}), (i,s,u)) =
                      if PointSet.size stones > PointSet.size s then
                        (id, stones, PointSet.union u s)
                      else
                        (i, s, PointSet.union u stones)

                  val i_s_u = (point, PointSet.empty, PointSet.singleton point)

                  val (id,_,update) = List.foldl find_id i_s_u friendly_blocks

                  fun update_id (p,m) = PointMap.insert m (p,id)

                  val black_stones =
                      PointSet.foldl update_id black_stones update
                in
                  (black_stones,id)
                end

            val black_blocks =
                let
                  fun del ((i,_),m) = PointMap.delete m i

                  val black_blocks = List.foldl del black_blocks friendly_blocks
                in
                  PointMap.insert black_blocks (block_id,block)
                end

            val black_blocks =
                if List.null captured_blocks then black_blocks
                else
                  PointMap.transform
                    (fn b => increaseLiberties b captured_stones) black_blocks
          in
            Board
              {fixed = fixed,
               emptyPoints = emptyPoints,
               stones = {black = black_stones, white = white_stones},
               blocks = {black = black_blocks, white = white_blocks}}
          end
      end;
in
  fun placeStone permitSuicide board side point =
      let
(*GomiTrace5
        val () = trace ("# Board.place: " ^ Side.toString side ^
                        " " ^ Point.toString point ^ "\n")
        val {black,white} = countStones board
        val () = trace ("# Board.place: #black = " ^ Int.toString black ^
                        ", #white = " ^ Int.toString white ^ "\n")
*)

        val board =
            case side of
              Side.Black => board
            | Side.White => switchSides board

        val board = placeBlack permitSuicide board point

        val board =
            case side of
              Side.Black => board
            | Side.White => switchSides board
      in
        board
      end
      handle Error err => raise Error ("Board.placeStone: " ^ err);
end;

fun tabulate dim peekFn =
    let
      fun processPoint (point,acc) =
          case peekFn point of
            SOME side => placeStone {permitSuicide = false} acc side point
          | NONE => acc
    in
      Dimensions.fold processPoint (empty dim) dim
    end;

fun clearPoints board set =
    let
(*GomiTrace5
      val _ = Print.trace PointSet.pp "Board.clearPoints: set" set
*)
    in
      if PointSet.subset set (emptyPoints board) then board
      else
        let
          fun pk p = if PointSet.member p set then NONE else destStone board p
        in
          tabulate (dimensions board) pk
        end
    end;

fun transform sym board =
    if Symmetry.isIdentity sym then board
    else
      let
        val sym = Symmetry.invert sym

        fun pk p =
            Option.map
              (Symmetry.transformSide sym)
              (destStone board (Symmetry.transformPoint sym p))
      in
        tabulate (dimensions board) pk
      end;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun pp board =
    let
      val {files,ranks} = dimensions board

      val rankSize = size (Point.rankToString (ranks - 1))

      fun paddingRankSize n =
          Print.ppString (nChars #" " (rankSize - n))

      fun paddedRank y =
          let
            val s = Point.rankToString y
          in
            Print.sequence (paddingRankSize (size s)) (Print.ppString s)
          end

      fun entry y x =
          if y = ~2 orelse y = ranks + 1 then
            if x = ~1 then paddingRankSize ~3
            else if x < files then
              Print.sequence
                (if x = 0 then Print.skip else Print.break)
                (Print.ppString (Point.fileToString x))
            else
              Print.skip
          else if y = ~1 orelse y = ranks then
            if x = ~1 then paddingRankSize ~1
            else if x = 0 then Print.ppString "+--"
            else if x < files then Print.ppString "--"
            else Print.ppString "-+"
          else
            if x = ~1 then
              Print.program
                [paddedRank y,
                 Print.break,
                 Print.ppString "|"]
            else if x < files then
              let
                val s =
                    case destStone board (Point.Point {file = x, rank = y}) of
                      NONE => "."
                    | SOME Side.Black => "#"
                    | SOME Side.White => "o"
              in
                Print.sequence Print.break (Print.ppString s)
              end
            else
              Print.program
                [Print.break,
                 Print.ppString "|",
                 Print.break,
                 Print.ppString (Point.rankToString y)]

      fun rank y =
          Print.sequence
            (Print.inconsistentBlock 2
               (List.map (entry y) (interval ~1 (files + 2))))
            Print.newline
    in
      Print.inconsistentBlock 0
        (List.map rank (List.rev (interval ~2 (ranks + 4))))
    end;

val toString = Print.toString pp;

end

structure BoardOrdered =
struct type t = Board.board val compare = Board.compare end

structure BoardMap = KeyMap (BoardOrdered);

structure BoardSet = ElementSet (BoardMap);
