(* ========================================================================= *)
(* DATABASE MAPPING PATTERNS TO FORMULA PROBABILITIES                        *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature Database =
sig

(* ------------------------------------------------------------------------- *)
(* A type of pattern set.                                                    *)
(* ------------------------------------------------------------------------- *)

type patterns = PatternSet.set

type patternsUpdate = {add : patterns, remove : patterns}

val noPatterns : patterns

val emptyPatternsUpdate : patternsUpdate

val updatePatterns : patterns -> patternsUpdate -> patterns

val subtractPatterns : patterns -> patterns -> patternsUpdate

(* ------------------------------------------------------------------------- *)
(* A type of estimated formula probabilities.                                *)
(* ------------------------------------------------------------------------- *)

type estimate = Probability.probability FormulaMap.map

type estimateUpdate = Probability.probability option FormulaMap.map

val emptyEstimate : estimate

val updateEstimate : estimate -> estimateUpdate -> estimate

val subtractEstimate : estimate -> estimate -> estimateUpdate

(* ------------------------------------------------------------------------- *)
(* A type of pattern database.                                               *)
(* ------------------------------------------------------------------------- *)

type database

val new : Position.parameters -> database

val size : database -> {patterns : int, connections : int, formulas : int}

val patterns : database -> patterns

val formulas : database -> FormulaSet.set

val estimate : database -> patterns -> estimate

val estimateAfter :
    database -> patterns -> estimate -> patternsUpdate -> estimateUpdate

val learn :
    database -> patterns -> (Formula.formula -> bool) ->
    database * patternsUpdate

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

val pp : database Print.pp

val toString : database -> string

end
