(* ========================================================================= *)
(* GO BOARD DIMENSIONS                                                       *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Dimensions :> Dimensions =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Go board dimensions.                                                      *)
(* ------------------------------------------------------------------------- *)

type dimensions = {files : int, ranks : int};

val default = {files = 19, ranks = 19};

fun files ({files = f, ...} : dimensions) = f;

fun ranks ({ranks = r, ...} : dimensions) = r;

fun numPoints ({files,ranks} : dimensions) = files * ranks;

fun member (Point.Point {file,rank}) {files,ranks} =
    0 <= file andalso file < files andalso
    0 <= rank andalso rank < ranks;

fun fold f acc {files,ranks} =
    let
      val maxFile = files - 1
      and maxRank = ranks - 1

      fun fold ~1 _ acc = acc
        | fold x ~1 acc = fold (x - 1) maxRank acc
        | fold x y acc =
          fold x (y - 1) (f (Point.Point {file = x, rank = y}, acc))
    in
      fold maxFile maxRank acc
    end;

local
  fun add (p,s) = PointSet.add s p;
in
  val points = fold add PointSet.empty;
end;

fun map f =
    let
      fun add (p,m) = PointMap.insert m (p, f p)
    in
      fold add (PointMap.new ())
    end;

fun app f dim =
    let
      fun g (point,()) = f point
    in
      fold g () dim
    end;

(* ------------------------------------------------------------------------- *)
(* Square boards.                                                            *)
(* ------------------------------------------------------------------------- *)

fun mkSquare n : dimensions = {files = n, ranks = n};

fun destSquare ({files,ranks} : dimensions) =
    if files = ranks then SOME files else NONE;

fun isSquare dim = Option.isSome (destSquare dim);

(* ------------------------------------------------------------------------- *)
(* Neighbouring points.                                                      *)
(* ------------------------------------------------------------------------- *)

fun neighbours {files,ranks} (Point.Point {file = x, rank = y}) =
    let
      fun add s (f,r) = PointSet.add s (Point.Point {file = f, rank = r})

      val s = PointSet.empty
      val s = if x = 0 then s else add s (x - 1, y)
      val s = if x = files - 1 then s else add s (x + 1, y)
      val s = if y = 0 then s else add s (x, y - 1)
      val s = if y = ranks - 1 then s else add s (x, y + 1)
    in
      s
    end;

fun foldNeighbours f acc dim p = PointSet.foldl f acc (neighbours dim p);

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun toString {files,ranks} = Int.toString files ^ "x" ^ Int.toString ranks;

val pp = Print.ppMap toString Print.ppString;

end
