(* ========================================================================= *)
(* THE GO PLAYING ENGINE                                                     *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Engine :> Engine =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Constants.                                                                *)
(* ------------------------------------------------------------------------- *)

val SAMPLE_GAMES_PER_MOVE = 0;

(* ------------------------------------------------------------------------- *)
(* A type of go engines.                                                     *)
(* ------------------------------------------------------------------------- *)

datatype engine =
    Engine of
      {fixed :
         {name : string,
          version : string},
       position : Position.position,
       database : Database.database};

fun new {name,version,position} =
    Engine
      {fixed =
         {name = name,
          version = version},
       position = position,
       database = Database.new (Position.parameters position)};

fun position (Engine {position = p, ...}) = p;

fun database (Engine {database = d, ...}) = d;

fun rules engine = Position.rules (position engine);

fun komi engine = Position.komi (position engine);

fun dimensions engine = Position.dimensions (position engine);

fun toMove engine = Position.toMove (position engine);

(* ------------------------------------------------------------------------- *)
(* Engine operations.                                                        *)
(* ------------------------------------------------------------------------- *)

fun reset engine parm =
    let
      val Engine {fixed,position,database} = engine

      val sameParm =
          Position.equalParameters parm (Position.parameters position)

      val position =
          if sameParm andalso Position.isInitial position then position
          else Position.initial parm

      val database = if sameParm then database else Database.new parm
    in
      Engine {fixed = fixed, position = position, database = database}
    end;

fun playMove engine move =
    let
      val Engine {fixed,position,database} = engine

      val position = Position.playMove position move
    in
      Engine {fixed = fixed, position = position, database = database}
    end;

fun setToMove engine side =
    if Side.equal (toMove engine) side then engine
    else playMove engine Move.Pass;

fun genmove engine =
    let
      val Engine {fixed,position,database} = engine

      val sample = ISample.new position database

      val database =
          funpow SAMPLE_GAMES_PER_MOVE (ISample.sample sample) database

      val move = raise Bug "Engine.genmove"

      val engine =
          Engine {fixed = fixed, position = position, database = database}
    in
      (move,engine)
    end;

(* ------------------------------------------------------------------------- *)
(* Go text protocol operations.                                              *)
(* ------------------------------------------------------------------------- *)

fun boardsize engine n =
    let
      val parm =
          {rules = rules engine,
           komi = komi engine,
           dimensions = Dimensions.mkSquare n}
    in
      reset engine parm
    end;

fun clearBoard engine =
    reset engine (Position.parameters (position engine));

fun finalScore engine =
    let
      val Engine {position,database,...} = engine
    in
      raise Bug "Engine.finalScore"
    end;

fun generateMove engine side =
    let
      val engine = setToMove engine side
      val (move,engine) = genmove engine
      val engine = playMove engine move
    in
      (move,engine)
    end;

fun komi engine komi =
    let
      val Engine {fixed,position,database} = engine
    in
      if Komi.equal komi (Position.komi position) then engine
      else
        let
          val rules = Position.rules position
          and board = Position.board position
          and toMove = Position.toMove position
          and prisoners = Position.prisoners position

          val position =
              Position.new
                {rules = rules,
                 komi = komi,
                 board = board,
                 toMove = toMove,
                 prisoners = prisoners}
        in
          Engine {fixed = fixed, position = position, database = database}
        end
    end;

fun name (Engine {fixed = {name = n, ...}, ...}) = n;

fun play engine side move = playMove (setToMove engine side) move;

fun quit (_ : engine) = ();

fun version (Engine {fixed = {version = v, ...}, ...}) = v;

end
