(* ========================================================================= *)
(* FORMULAS OVER TERMINAL GO POSITIONS                                       *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature Formula =
sig

(* ------------------------------------------------------------------------- *)
(* Integers associated with terminal go positions.                           *)
(* ------------------------------------------------------------------------- *)

datatype integer =
  (* Primitive integer operations *)
    Integer of int
  | Negate of integer
  | Add of integer * integer
  | Multiply of integer * integer

  (* The number of black territory points minus the number of white points *)
  | PointsScore

  (* The number of territory points in a group *)
  | PointsGroup of Point.point

val compareInteger : integer * integer -> order

val isValidInteger : Dimensions.dimensions -> integer -> bool

val transformInteger : Symmetry.symmetry -> integer -> integer

(* ------------------------------------------------------------------------- *)
(* Point status in terminal go positions.                                    *)
(* ------------------------------------------------------------------------- *)

datatype status =
  (* Primitive status operations *)
    Status of Status.status

  (* Status of an individual point *)
  | StatusPoint of Point.point

val compareStatus : status * status -> order

val isValidStatus : Dimensions.dimensions -> status -> bool

val transformStatus : Symmetry.symmetry -> status -> status

(* ------------------------------------------------------------------------- *)
(* Sides in terminal go positions.                                           *)
(* ------------------------------------------------------------------------- *)

datatype side =
  (* Primitive status operations *)
    Side of Side.side option
  | Opponent of side

  (* Side of a point status *)
  | SideStatus of status
  | SideStoneStatus of status
  | SideEyeStatus of status

val compareSide : side * side -> order

val isValidSide : Dimensions.dimensions -> side -> bool

val transformSide : Symmetry.symmetry -> side -> side

val blackSide : side

val whiteSide : side

val noSide : side

(* ------------------------------------------------------------------------- *)
(* Formulas over terminal go positions.                                      *)
(* ------------------------------------------------------------------------- *)

datatype formula =
  (* Primitive logical operations *)
    Boolean of bool
  | Not of formula
  | And of formula * formula
  | Or of formula * formula
  | Implies of formula * formula
  | Iff of formula * formula

  (* Integer formulas *)
  | LessThan of integer * integer
  | LessEqual of integer * integer
  | IntegerEqual of integer * integer
  | GreaterEqual of integer * integer
  | GreaterThan of integer * integer

  (* Status formulas *)
  | StatusEqual of status * status
  | StatusMember of status * StatusSet.set

  (* Side formulas *)
  | SideEqual of side * side

  (* Group formulas *)
  | ConnectedGroup of Point.point * Point.point
  | SekiGroup of Point.point

val compare : formula * formula -> order

val isValid : Dimensions.dimensions -> formula -> bool

val transform : Symmetry.symmetry -> formula -> formula

val alwaysTrue : formula

val alwaysFalse : formula

(* ------------------------------------------------------------------------- *)
(* Game result.                                                              *)
(* ------------------------------------------------------------------------- *)

val isBlackWin : Komi.komi -> formula

val isWhiteWin : Komi.komi -> formula

val isSideWin : Komi.komi -> Side.side -> formula

val isWin : Komi.komi -> formula

val isDraw : Komi.komi -> formula

(* ------------------------------------------------------------------------- *)
(* Individual point status.                                                  *)
(* ------------------------------------------------------------------------- *)

val isBlackStone : Point.point -> formula

val isWhiteStone : Point.point -> formula

val isSideStone : Side.side -> Point.point -> formula

val isStone : Point.point -> formula

val isBlackEye : Point.point -> formula

val isWhiteEye : Point.point -> formula

val isSideEye : Side.side -> Point.point -> formula

val isEye : Point.point -> formula

val isBlackTerritory : Point.point -> formula

val isWhiteTerritory : Point.point -> formula

val isSideTerritory : Side.side -> Point.point -> formula

val isTerritory : Point.point -> formula

val isSekiPoint : Point.point -> formula

end
