(* ========================================================================= *)
(* GNU TEXT PROTOCOL                                                         *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure GTP :> GTP =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* A type of GTP engines.                                                    *)
(* ------------------------------------------------------------------------- *)

type gtp = Engine.engine;

fun initial {name,version,rules,komi,boardsize} =
    let
      val position =
          Position.initialSquare
            {rules = rules,
             komi = komi,
             boardsize = boardsize}
    in
      Engine.new {name = name, version = version, position = position}
    end;

(* ------------------------------------------------------------------------- *)
(* Responses to commands.                                                    *)
(* ------------------------------------------------------------------------- *)

datatype response =
    Response of
      {success : bool,
       message : string option,
       gtp : gtp option};

val success = Response {success = true, message = NONE, gtp = NONE};

fun success_ack gtp =
    Response {success = true, message = NONE, gtp = SOME gtp};

fun success_message mesg =
    Response {success = true, message = SOME mesg, gtp = NONE};

fun success_message_ack mesg gtp =
    Response {success = true, message = SOME mesg, gtp = SOME gtp};

fun failure_message mesg =
    Response {success = false, message = SOME mesg, gtp = NONE};

(* ------------------------------------------------------------------------- *)
(* Commands.                                                                 *)
(* ------------------------------------------------------------------------- *)

val COMMANDS =
    ["boardsize",
     "clear_board",
     "final_score",
     "genmove",
     "komi",
     "known_command",
     "list_commands",
     "name",
     "play",
     "protocol_version",
     "quit",
     "version"];

fun preprocess_command cmd =
    let
      fun remove_ctrl #" " = " "
        | remove_ctrl #"\t" = " "
        | remove_ctrl #"\n" = "\n"
        | remove_ctrl c = if Char.isGraph c then str c else ""

      fun remove_comments "" = ""
        | remove_comments s = hd (String.tokens (equal #"#") s)

      val cmd = String.translate remove_ctrl cmd
      val lines = String.tokens (equal #"\n") cmd
      val lines = List.map remove_comments lines
    in
      trim (join " " lines)
    end;

fun process_command gtp cmd_args =
    (case cmd_args of
       ("boardsize",[n]) =>
       (case Int.fromString n of
          NONE => failure_message "syntax error"
        | SOME n =>
          if n <= 0 then failure_message "unacceptable size"
          else success_ack (Engine.boardsize gtp n))
     | ("clear_board",[]) => success_ack (Engine.clearBoard gtp)
     | ("final_score",[]) =>
       success_message (Score.toString (Engine.finalScore gtp))
     | ("genmove",[side]) =>
       (case total Side.fromString side of
          NONE => failure_message "syntax error"
        | SOME side =>
          let
            val (move,gtp) = Engine.generateMove gtp side
          in
            success_message_ack (Move.toString move) gtp
          end)
     | ("komi",[k]) =>
       (case Real.fromString k of
          NONE => failure_message "syntax error"
        | SOME r => success_ack (Engine.komi gtp r))
     | ("known_command",[cmd]) =>
       success_message (if mem cmd COMMANDS then "true" else "false")
     | ("list_commands",[]) => success_message (join "\n" COMMANDS)
     | ("name",[]) => success_message (Engine.name gtp)
     | ("play", [side,move]) =>
       (case total Side.fromString side of
          NONE => failure_message "syntax error"
        | SOME side =>
          case total Move.fromString move of
            NONE => failure_message "syntax error"
          | SOME move =>
            case total (Engine.play gtp side) move of
              NONE => failure_message "illegal_move"
            | SOME gtp => success_ack gtp)
     | ("protocol_version",[]) => success_message "2"
     | ("quit",[]) => (Engine.quit gtp; success)
     | ("version",[]) => success_message (Engine.version gtp)
     | (cmd,_) =>
       if mem cmd COMMANDS then failure_message "syntax error"
       else failure_message "unknown command")
    handle Error err => failure_message err;

fun command gtp cmd =
    let
      val cmd_args as (cmd,args) =
          case preprocess_command cmd of
            "" => ("",[])
          | s => hdTl (String.tokens Char.isSpace s)

      val (id, cmd_args as (cmd,args)) =
          if not (List.all Char.isDigit (String.explode cmd)) then
            (NONE,cmd_args)
          else
            (SOME cmd, case args of [] => ("",[]) | h :: t => (h,t))

      val Response {success, message, gtp = gtp'} =
          if cmd <> "" then process_command gtp cmd_args
          else Response {success = true, message = NONE, gtp = NONE}

      val response =
          (if success then "=" else "?") ^
          (case id of NONE => "" | SOME s => s) ^
          (case message of NONE => "" | SOME s => " " ^ s) ^ "\n\n"

      val gtp = if cmd = "quit" then NONE else SOME (Option.getOpt (gtp',gtp))
    in
      (response,gtp)
    end;

fun loop {input,output} =
    let
      fun process gtp =
          case TextIO.inputLine input of
            NONE => ()
          | SOME cmd =>
            let
              val (response,gtp) = command gtp cmd
              val () = TextIO.output (output,response)
              val () = TextIO.flushOut output
            in
              case gtp of NONE => () | SOME gtp => process gtp
            end
    in
      process
    end;

end
