(* ========================================================================= *)
(* GO GAMES                                                                  *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Game :> Game =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Chatting                                                                  *)
(* ------------------------------------------------------------------------- *)

val module = "Game";
fun chatting l = tracing {module = module, level = l};
fun chat s = (trace s; true);

fun chat_pp pref pp_x x =
    let
      val pp = Parser.ppBinop " =" Parser.ppString pp_x
    in
      chat (Parser.toString pp (pref,x) ^ "\n")
    end;

(* ------------------------------------------------------------------------- *)
(* A type of Go games.                                                       *)
(* ------------------------------------------------------------------------- *)

datatype game =
    Game of
      {tags : string StringMap.map,
       position : Position.position,
       history : (Position.position * Position.move) list};

fun new () =
    Game
      {tags = StringMap.new (),
       position = Position.new {files = !Globals.boardSize, ranks = !Globals.boardSize},
       history = []};

fun position (Game {position = p, ...}) = p;

fun history (Game {history = h, ...}) = h;

fun toMove game = Position.toMove (position game);

fun moves game = map snd (history game);

(* ------------------------------------------------------------------------- *)
(* Tags.                                                                     *)
(* ------------------------------------------------------------------------- *)

type tag = {name : string, value : string}

fun tags (Game {tags = t, ...}) =
    StringMap.foldl (fn (n,v,l) => {name = n, value = v} :: l) [] t;

fun tag (Game {tags = t, ...}) n = StringMap.peek t n;

fun insertTag game tag =
    let
      val Game {tags,position,history} = game
      and {name,value} = tag
      val tags = StringMap.insert tags (name,value)
    in
      Game {tags = tags, position = position, history = history}
    end;

(* ------------------------------------------------------------------------- *)
(* Scoring.                                                                  *)
(* ------------------------------------------------------------------------- *)

fun score :
val score : game -> real


(* ------------------------------------------------------------------------- *)
(* Positions                                                                 *)
(* ------------------------------------------------------------------------- *)

datatype position =
    Position of
      {board : Board.board,
       toMove : Side.side,
       history : BoardSet.set};

fun new dim =
    Position
      {board = Board.new dim,
       toMove = Side.Black,
       history = BoardSet.empty};

fun board (Position {board = b, ...}) = b;

fun toMove (Position {toMove = s, ...}) = s;

(* ------------------------------------------------------------------------- *)
(* Legal moves.                                                              *)
(* ------------------------------------------------------------------------- *)

datatype move = Stone of Point.point | Pass;

fun moveToString Pass = "pass"
  | moveToString (Stone point) = Point.toString point;

fun moveFromString move =
    case String.map Char.toUpper move of
      "PASS" => Pass
    | move => Stone (Point.fromString move);

fun move (Position {board,toMove,history}) Pass =
    Position {board = board, toMove = Side.opponent toMove, history = history}
  | move (Position {board,toMove,history}) (Stone point) =
    let
      val history = BoardSet.add history board

      val board = Board.place board toMove point

      val toMove = Side.opponent toMove

      val _ = not (BoardSet.member board history) orelse
              raise Error "Position.move: forbidden by positional superko"
    in
      Position {board = board, toMove = toMove, history = history}
    end;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun toString (Position {board,...}) = Board.toString board;

end
