(* ========================================================================= *)
(* THE SMART GAME FILE FORMAT (SGF v4) FOR THE GAME OF GO                    *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure GoSgf :> GoSgf =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Helper functions.                                                         *)
(* ------------------------------------------------------------------------- *)

local
  val offsetLower = Char.ord #"a"
  and offsetUpper = Char.ord #"A" - 26;
in
  fun coordToChar c =
      let
(*GomiDebug
        val _ = 0 <= c orelse raise Bug "coordToChar: negative"
        val _ = c < 52 orelse raise Bug "coordToChar: large"
*)
      in
        if c < 26 then Char.chr (c + offsetLower)
        else Char.chr (c + offsetUpper)
      end;

  fun coordFromChar c =
      if Char.isLower c then SOME (Char.ord c - offsetLower)
      else if Char.isUpper c then SOME (Char.ord c - offsetUpper)
      else NONE;
end;

(* ------------------------------------------------------------------------- *)
(* Go SGF types.                                                             *)
(* ------------------------------------------------------------------------- *)

type point = Point.point;

type move = Move.move;

type stone = Point.point;

type properties = (point,move,stone) Sgf.properties

type node = (point,move,stone) Sgf.node;

type game = (point,move,stone) Sgf.game;

type collection = (point,move,stone) Sgf.collection;

(* ------------------------------------------------------------------------- *)
(* SGF properties.                                                           *)
(* ------------------------------------------------------------------------- *)

val MOVE_PROPERTIES = Sgf.MOVE_PROPERTIES;

val SETUP_PROPERTIES = Sgf.SETUP_PROPERTIES;

fun sgfDimensions (files,ranks) : Dimensions.dimensions =
    {files = files, ranks = ranks};

fun sgfPoint ({ranks,...} : Dimensions.dimensions) (Point.Point {file,rank}) =
    Point.Point {file = file, rank = (ranks - 1) - rank};

fun sgfMove dim (Move.Stone point) =
    let
      val isDeprecatedPass =
          case point of
            Point.Point {file = 19, rank = 19} =>
            let
              val {files,ranks} = dim
            in
              files <= 19 andalso ranks <= 19
            end
          | _ => false
    in
      if isDeprecatedPass then Move.Pass
      else Move.Stone (sgfPoint dim point)
    end
  | sgfMove _ move = move;

fun sgfPointRectangle dim (p1,p2) =
    PointSet.mkRectangle (sgfPoint dim p1, sgfPoint dim p2);

fun sgfPointList dim l = PointSet.unionList (List.map (sgfPointRectangle dim) l);

fun peekAB dim props = Option.map (sgfPointList dim) (Sgf.peekAB props);

fun peekAE dim props = Option.map (sgfPointList dim) (Sgf.peekAE props);

fun peekAW dim props = Option.map (sgfPointList dim) (Sgf.peekAW props);

fun peekB dim props = Option.map (sgfMove dim) (Sgf.peekB props);

fun peekPL props = Sgf.peekPL props;

fun peekSZ props = Option.map sgfDimensions (Sgf.peekSZ props);

fun peekW dim props = Option.map (sgfMove dim) (Sgf.peekW props);

(* ------------------------------------------------------------------------- *)
(* Pretty-printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun ppPoint (Point.Point {file,rank}) =
    Print.sequence
      (Print.ppChar (coordToChar file))
      (Print.ppChar (coordToChar rank));

fun ppMove (Move.Stone pt) = ppPoint pt
  | ppMove Move.Pass = Print.skip;

val ppStone = ppPoint;

(* ------------------------------------------------------------------------- *)
(* Parsing.                                                                  *)
(* ------------------------------------------------------------------------- *)

type propertyParser = (point,move,stone) Sgf.propertyParser;

local
  infixr 8 ++
  infixr 7 >>
  infixr 6 ||

  open Parse;

  val space = many (some Char.isSpace) >> K ();

  val coordParser = maybe coordFromChar;
in
  val pointParser =
      coordParser ++ coordParser >>
      (fn (file,rank) => Point.Point {file = file, rank = rank});

  val moveParser =
      pointParser >> Move.Stone ||
      nothing >> K Move.Pass;

  val stoneParser = pointParser;
end;

val propertyParser =
    (* Properties with type "-" *)
    fn "TB" => Sgf.pointEListParser pointParser
     | "TW" => Sgf.pointEListParser pointParser

    (* Properties with type "game-info" *)
     | "HA" => Sgf.singleParser Sgf.numberParser
     | "KM" => Sgf.singleParser Sgf.realParser

    (* Back off to general properties *)
     | propId =>
       Sgf.generalPropertyParser
         {pointParser = pointParser,
          moveParser = moveParser,
          stoneParser = stoneParser} propId;

(* ------------------------------------------------------------------------- *)
(* Reading from a file.                                                      *)
(* ------------------------------------------------------------------------- *)

fun read {filename} =
    Sgf.read {filename = filename, propertyParser = propertyParser};

(* ------------------------------------------------------------------------- *)
(* Writing to a file.                                                        *)
(* ------------------------------------------------------------------------- *)

fun write {collection,filename} =
    Sgf.write
      {ppPoint = ppPoint,
       ppMove = ppMove,
       ppStone = ppStone,
       collection = collection,
       filename = filename};

end
