(* ========================================================================= *)
(* IMPERATIVE GO BOARDS                                                      *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature IBoard =
sig

(* ------------------------------------------------------------------------- *)
(* Imperative go boards.                                                     *)
(* ------------------------------------------------------------------------- *)

type board

val equal : board -> board -> bool

val clone : board -> board

val copy : board -> board -> unit

val peek : board -> IPoint.point -> Side.side option

val empty : board -> IIntSet.set

val toBoard : board -> Board.board

val fromBoard : Board.board -> board

(* ------------------------------------------------------------------------- *)
(* Blocks.                                                                   *)
(* ------------------------------------------------------------------------- *)

type block

val equalBlock : block -> block -> bool

val block : board -> IPoint.point -> block

val stones : board -> block -> int

val ledges : board -> block -> int

val ppBlock : block Print.pp

val toStringBlock : block -> string

(* ------------------------------------------------------------------------- *)
(* Moves.                                                                    *)
(* ------------------------------------------------------------------------- *)

(* Always succeeds *)

val playPassMove : board -> unit

(* These return a success result *)

val playStoneMove : board -> Side.side -> IPoint.point -> bool

val playSensibleStoneMove : board -> Side.side -> IPoint.point -> bool

(* These are assumed to succeed *)

val playMove : board -> Side.side -> IMove.move -> unit

val playMoves : board -> Side.side -> IMove.move list -> unit

(* ------------------------------------------------------------------------- *)
(* Last move played.                                                         *)
(* ------------------------------------------------------------------------- *)

val lastMove : board -> IMove.move option

val lastMoveCaptured : board -> IPoint.point IStack.stack

val undoLastMove : board -> unit

(* ------------------------------------------------------------------------- *)
(* Filling one-point eyes at the end of the game.                            *)
(* ------------------------------------------------------------------------- *)

(* Returns the side owning the one-point eye if it is one *)

val fillEye : board -> IPoint.point -> Side.side option

(* ------------------------------------------------------------------------- *)
(* Pattern matching.                                                         *)
(* ------------------------------------------------------------------------- *)

val matchesInteger : board -> Pattern.integer -> int

val matchesSide :
    board -> {toMove : Side.side} -> Pattern.side -> Side.side option

val matches : board -> {toMove : Side.side} -> Pattern.pattern -> bool

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

val pp : board Print.pp

val toString : board -> string

end
