(* ========================================================================= *)
(* IMPERATIVE GO BOARDS                                                      *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure IBoard :> IBoard =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Constants.                                                                *)
(* ------------------------------------------------------------------------- *)

val EDGE_POINT = ~1;
val EMPTY_POINT = EDGE_POINT - 1;
val MARKED_POINT = EMPTY_POINT - 1;
val MIN_POINT = MARKED_POINT;

val EDGE_BLOCK = ~1;
val EMPTY_BLOCK = EDGE_BLOCK - 1;
val MARKED_BLOCK = EMPTY_BLOCK - 1;
val DISAPPEARING_BLOCK = MARKED_BLOCK - 1;
val BEING_MARKED_BLOCK = DISAPPEARING_BLOCK - 1;

(*GomiDebug
val PLAY_SENSIBLE_STONE_MOVE_CHECK = 100;
val UNDO_LAST_MOVE_CHECK = 100;
*)

(* ------------------------------------------------------------------------- *)
(* Helper functions.                                                         *)
(* ------------------------------------------------------------------------- *)

fun increment a i (x : int) = Array.update (a, i, Array.sub (a,i) + x);

fun decrement a i x = increment a i (~x);

(* ------------------------------------------------------------------------- *)
(* Imperative blocks of stones.                                              *)
(* ------------------------------------------------------------------------- *)

type block = IPoint.point;

fun equalBlock (b1 : block) b2 = IPoint.equal b1 b2;

fun toStringBlock block = "[" ^ IPoint.toString block ^ "]";

val ppBlock = Print.ppMap toStringBlock Print.ppString;

fun isEmptyPoint blockLedges p = Array.sub (blockLedges,p) = EMPTY_POINT;

local
  fun chaseBlock blockLedges p pp =
      let
        val ppp = Array.sub (blockLedges,pp)
(*GomiDebug
        val _ = ppp <> EDGE_POINT orelse raise Bug "edge block"
*)
      in
        if ppp = EMPTY_POINT then DISAPPEARING_BLOCK
        else if ppp = MARKED_POINT then BEING_MARKED_BLOCK
        else if ppp < 0 then pp
        else
          let
            val () = Array.update (blockLedges,p,ppp)
          in
            chaseBlock blockLedges pp ppp
          end
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.chaseBlock: " ^ bug)
           | e => raise Bug ("IBoard.chaseBlock: " ^ exnMessage e);
*)
in
  fun peekBlock blockLedges p =
      let
        val pp = Array.sub (blockLedges,p)
      in
        if pp = EDGE_POINT then EDGE_BLOCK
        else if pp = EMPTY_POINT then EMPTY_BLOCK
        else if pp = MARKED_POINT then MARKED_BLOCK
        else if pp < 0 then p
        else chaseBlock blockLedges p pp
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.peekBlock: " ^ bug)
           | e => raise Bug ("IBoard.peekBlock: " ^ exnMessage e);
*)
end;

fun markStone blockLedges block point =
    let
      val () = Array.update (blockLedges,point,MARKED_POINT)
      val () = markBlock blockLedges block (IPoint.moveUp point)
      val () = markBlock blockLedges block (IPoint.moveDown point)
      val () = markBlock blockLedges block (IPoint.moveLeft point)
      val () = markBlock blockLedges block (IPoint.moveRight point)
    in
      ()
    end

and markBlock blockLedges block point =
    let
      val block' = peekBlock blockLedges point
    in
      if block' = block orelse block' = BEING_MARKED_BLOCK then
        markStone blockLedges block point
      else
        ()
    end;

(* ------------------------------------------------------------------------- *)
(* Liberty edges (a.k.a. ledges).                                            *)
(* ------------------------------------------------------------------------- *)

local
  val ZERO_LEDGES = MIN_POINT - 1;

  val EDGE_TO_ZERO = ZERO_LEDGES - EDGE_POINT;

  fun packLedges n =
      let
(*GomiDebug
        val _ = n >= 0 orelse
                raise Bug "IBoard.packLedges: nonpositive ledge"
*)
      in
        ZERO_LEDGES - n
      end;

  fun unpackLedges n =
      let
(*GomiDebug
        val _ = n <= ZERO_LEDGES orelse
                raise Bug "IBoard.unpackLedges: bad packed ledge"
*)
      in
        ZERO_LEDGES - n
      end;
in
  fun findLedges blockLedges block =
      unpackLedges (Array.sub (blockLedges,block));

  fun incrementLedges blockLedges block x = decrement blockLedges block x;

  fun decrementLedges blockLedges block x = increment blockLedges block x;

  fun storeLedges blockLedges block ledges =
      Array.update (blockLedges, block, packLedges ledges);

  fun edgeToZeroLedges blockLedges block =
      increment blockLedges block EDGE_TO_ZERO;
end;

(* ------------------------------------------------------------------------- *)
(* Side of and number of stones in a block.                                  *)
(* ------------------------------------------------------------------------- *)

fun findSide sideStones block =
    let
      val n = Array.sub (sideStones,block)
(*GomiDebug
      val _ = n <> 0 orelse
              raise Bug "IBoard.findSide: illegal entry"
*)
    in
      if n > 0 then Side.Black else Side.White
    end;

fun findStones sideStones block =
    let
      val n = Array.sub (sideStones,block)
(*GomiDebug
      val _ = n <> 0 orelse
              raise Bug "IBoard.findStones: illegal entry"
*)
    in
      Int.abs n
    end;

fun singleStone sideStones block = findStones sideStones block = 1;

fun multipleStones sideStones block = findStones sideStones block > 1;

fun incrementStones sideStones block x =
    let
      val n = Array.sub (sideStones,block)
      val n' = if n > 0 then n + x else n - x
(*GomiDebug
      val _ = n <> 0 orelse raise Bug "illegal existing entry"
      val _ = not (n > 0) orelse n' > 0 orelse
              raise Bug "positive to nonpositive"
      val _ = not (n < 0) orelse n' < 0 orelse
              raise Bug "negative to nonnegative"
*)
    in
      Array.update (sideStones,block,n')
    end
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.incrementStones:\n" ^ bug);
*)

fun decrementStones sideStones block x =
    incrementStones sideStones block (~x)
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.decrementStones:\n" ^ bug);
*)

fun incrementStonesSide sideStones side block x =
    let
      val n = Array.sub (sideStones,block)
(*GomiDebug
      val _ = n = 0 orelse n > 0 = Side.isBlack side orelse
              raise Bug "IBoard.incrementStones: entry is wrong side"
      val _ = x > 0 orelse
              raise Bug "IBoard.incrementStonesSide: nonpositive increment"
*)
      val n = case side of Side.Black => n + x | Side.White => n - x
    in
      Array.update (sideStones,block,n)
    end;

fun storeSideStones sideStones block side n =
    let
(*GomiDebug
      val _ = n <> 0 orelse
              raise Bug "IBoard.storeSideStones: nonpositive number"
*)
    in
      case side of
        Side.Black => Array.update (sideStones,block,n)
      | Side.White => Array.update (sideStones, block, ~n)
    end;

(* ------------------------------------------------------------------------- *)
(* Capturing blocks.                                                         *)
(* ------------------------------------------------------------------------- *)

local
  fun capture blockLedges captured block point =
      let
        val block' = peekBlock blockLedges point
      in
        if block' = EMPTY_BLOCK orelse block' = EDGE_BLOCK then
          ()
        else if block' = block orelse block' = DISAPPEARING_BLOCK then
          captureStone blockLedges captured block point
        else
          incrementLedges blockLedges block' 1
      end

  and captureStone blockLedges captured block point =
      let
        val () = IStack.push captured point
        val () = Array.update (blockLedges,point,EMPTY_POINT)
        val () = capture blockLedges captured block (IPoint.moveUp point)
        val () = capture blockLedges captured block (IPoint.moveDown point)
        val () = capture blockLedges captured block (IPoint.moveLeft point)
        val () = capture blockLedges captured block (IPoint.moveRight point)
      in
        ()
      end;
in
  fun captureBlock blockLedges sideStones captured block =
      let
(*GomiDebug
        val blockStones = findStones sideStones block
        val sizeCaptured = IStack.size captured
*)
        val () = captureStone blockLedges captured block block
(*GomiDebug
        val _ = IStack.size captured - sizeCaptured = blockStones orelse
                raise Bug "captured wrong number of stones"
*)
      in
        ()
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.capture: " ^ bug);
*)
end;

(* ------------------------------------------------------------------------- *)
(* Merging blocks.                                                           *)
(* ------------------------------------------------------------------------- *)

local
  fun unionBlocks blockLedges sideStones lastMoveBlocks block1 block2 =
    let
(*GomiDebug
          val _ = block1 <> block2 orelse
                  raise Bug "unionBlocks: same block"
*)
        val s1 = findStones sideStones block1
        and s2 = findStones sideStones block2
      in
        if s1 >= s2 then
          let
            val block2Ledges = findLedges blockLedges block2
            val () = incrementLedges blockLedges block1 block2Ledges
            val () = incrementStones sideStones block1 s2
            val () = Array.update (blockLedges,block2,block1)
            val () = IStack.push lastMoveBlocks block2
          in
            block1
          end
        else
          let
            val block1Ledges = findLedges blockLedges block1
            val () = incrementLedges blockLedges block2 block1Ledges
            val () = incrementStones sideStones block2 s1
            val () = Array.update (blockLedges,block1,block2)
            val () = IStack.push lastMoveBlocks block1
          in
            block2
          end
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.unionBlocks:\n" ^ bug);
*)

  fun unionThreeBlocks blockLedges sideStones lastMoveBlocks b1 b2 b3 =
      let
(*GomiDebug
          val _ = (b1 <> b2 andalso b1 <> b3 andalso b2 <> b3) orelse
                  raise Bug "unionThreeBlocks: duplicate block"
*)
        val b12 = unionBlocks blockLedges sideStones lastMoveBlocks b1 b2
      in
        unionBlocks blockLedges sideStones lastMoveBlocks b12 b3
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.unionThreeBlocks:\n" ^ bug);
*)

  fun unionFourBlocks blockLedges sideStones lastMoveBlocks b1 b2 b3 b4 =
      let
(*GomiDebug
          val _ = (b1 <> b2 andalso b1 <> b3 andalso b2 <> b3 andalso
                   b1 <> b4 andalso b2 <> b4 andalso b3 <> b4) orelse
                  raise Bug "unionFourBlocks: duplicate block"
*)
        val b12 = unionBlocks blockLedges sideStones lastMoveBlocks b1 b2
        and b34 = unionBlocks blockLedges sideStones lastMoveBlocks b3 b4
      in
        unionBlocks blockLedges sideStones lastMoveBlocks b12 b34
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.unionFourBlocks:\n" ^ bug);
*)

in
  fun extendBlock blockLedges sideStones index block ledges =
      let
        val () = incrementLedges blockLedges block ledges
        val () = incrementStones sideStones block 1
        val () = Array.update (blockLedges,index,block)
      in
        ()
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.extendBlock:\n" ^ bug);
*)

  fun connectTwoBlocks blockLedges sideStones lastMoveBlocks index b1 b2 ledges =
      let
        val b = unionBlocks blockLedges sideStones lastMoveBlocks b1 b2
      in
        extendBlock blockLedges sideStones index b ledges
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.connectTwoBlocks:\n" ^ bug);
*)

  fun connectThreeBlocks blockLedges sideStones lastMoveBlocks
                         index b1 b2 b3 ledges =
      let
        val b = unionThreeBlocks blockLedges sideStones lastMoveBlocks b1 b2 b3
      in
        extendBlock blockLedges sideStones index b ledges
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.connectThreeBlocks:\n" ^ bug);
*)

  fun connectFourBlocks blockLedges sideStones lastMoveBlocks index b1 b2 b3 b4 =
      let
        val b = unionFourBlocks blockLedges sideStones lastMoveBlocks b1 b2 b3 b4
      in
        extendBlock blockLedges sideStones index b ~4
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.connectFourBlocks:\n" ^ bug);
*)

end;

(* ------------------------------------------------------------------------- *)
(* Last move.                                                                *)
(* ------------------------------------------------------------------------- *)

datatype lastMoveStatus =
    UnknownLastMove
  | PassLastMove
  | StoneLastMove;

datatype lastMove =
    LastMove of
      {status : lastMoveStatus ref,
       point : IPoint.point ref,
       blocks : IPoint.point IStack.stack,
       captured : IPoint.point IStack.stack,
       ko : IPoint.point ref};

fun toStringStatus status =
    case status of
      UnknownLastMove => "Unknown"
    | PassLastMove => "Pass"
    | StoneLastMove => "Stone";

fun newLastMove () =
    let
      val status = ref UnknownLastMove

      and point = ref IPoint.NO_POINT

      and blocks = IStack.empty {maxSize = 4, defaultItem = IPoint.NO_POINT}

      and captured = IStack.empty {maxSize = IPoint.POINTS,
                                   defaultItem = IPoint.NO_POINT}

      and ko = ref IPoint.NO_POINT
    in
      LastMove
        {status = status,
         point = point,
         blocks = blocks,
         captured = captured,
         ko = ko}
    end;

fun cloneLastMove lastMove =
    let
      val LastMove
            {status = ref status,
             point = ref point,
             blocks,
             captured,
             ko = ref ko} = lastMove
    in
      LastMove
        {status = ref status,
         point = ref point,
         blocks = IStack.clone blocks,
         captured = IStack.clone captured,
         ko = ref ko}
    end;

fun copyLastMove src dst =
    let
      val LastMove
            {status = ref srcStatus,
             point = ref srcPoint,
             blocks = srcBlocks,
             captured = srcCaptured,
             ko = ref srcKo} = src

      and LastMove
            {status = dstStatus,
             point = dstPoint,
             blocks = dstBlocks,
             captured = dstCaptured,
             ko = dstKo} = dst

      val () = dstStatus := srcStatus
      and () = dstPoint := srcPoint
      and () = IStack.copy srcBlocks dstBlocks
      and () = IStack.copy srcCaptured dstCaptured
      and () = dstKo := srcKo
    in
      ()
    end;

fun knownLastMove (LastMove {status = ref status, ...}) =
    case status of
      UnknownLastMove => false
    | _ => true;

(* ------------------------------------------------------------------------- *)
(* Imperative go boards.                                                     *)
(* ------------------------------------------------------------------------- *)

datatype board =
    Board of
      {blockLedges : int Array.array,
       sideStones : int Array.array,
       empty : IIntSet.set,
       ko : IPoint.point ref,
       lastMove : lastMove};

fun empty (Board {empty = e, ...}) = e;

fun clone board =
    let
      val Board
            {blockLedges,
             sideStones,
             empty,
             ko = ref ko,
             lastMove} = board
    in
      Board
        {blockLedges = cloneArray blockLedges,
         sideStones = cloneArray sideStones,
         empty = IIntSet.clone empty,
         ko = ref ko,
         lastMove = cloneLastMove lastMove}
    end;

fun copy src dst =
    let
      val Board
            {blockLedges = srcBlockLedges,
             sideStones = srcSideStones,
             empty = srcEmpty,
             ko = ref srcKo,
             lastMove = srcLastMove} = src

      and Board
            {blockLedges = dstBlockLedges,
             sideStones = dstSideStones,
             empty = dstEmpty,
             ko = dstKo,
             lastMove = dstLastMove} = dst

      val () = Array.copy {src = srcBlockLedges, dst = dstBlockLedges, di = 0}
      val () = Array.copy {src = srcSideStones, dst = dstSideStones, di = 0}
      val () = IIntSet.copy srcEmpty dstEmpty
      val () = dstKo := srcKo
      val () = copyLastMove srcLastMove dstLastMove
    in
      ()
    end;

fun peek (Board {blockLedges,sideStones,...}) point =
    let
(*GomiDebug
      val _ = IPoint.onBoard point orelse raise Bug "edge point"
*)
      val block = peekBlock blockLedges point
(*GomiDebug
      val _ = block <> EDGE_BLOCK orelse raise Bug "edge block"
      val _ = block <> DISAPPEARING_BLOCK orelse raise Bug "disappearing block"
*)
    in
      if block = EMPTY_BLOCK then NONE else SOME (findSide sideStones block)
    end
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.peek: " ^ bug);
*)

fun equal board1 board2 =
    let
      fun eq p = peek board1 p = peek board2 p
    in
      List.all eq IPoint.boardPoints
    end
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.equal: " ^ bug);
*)

(* ------------------------------------------------------------------------- *)
(* Check board invariants.                                                   *)
(* ------------------------------------------------------------------------- *)

(*GomiDebug
fun check board =
    let
      val Board
            {blockLedges,
             sideStones,
             empty,
             ko = ref ko,
             lastMove} = board

      val LastMove
            {captured = lastMoveCaptured,
             ...} = lastMove

      fun isEdge point =
          let
            val edge = IPoint.isEdge point
            val edge' = Array.sub (blockLedges,point) = EDGE_POINT
            val _ = edge = edge' orelse raise Error "edge disagreement"
          in
            edge
          end

      val isEmpty = isEmptyPoint blockLedges

      fun chaseRoot point =
          let
            val p = Array.sub (blockLedges,point)
          in
            if p = EDGE_POINT then Right ("edge",[])
            else if p = EMPTY_POINT then Right ("empty",[])
            else if p = MARKED_POINT then Right ("marked",[])
            else if p < 0 then Left (point,0)
            else
              case chaseRoot p of
                Left (block,steps) => Left (block, steps + 1)
              | Right (err,ps) => Right (err, p :: ps)
          end

      fun getRoot point =
          case chaseRoot point of
            Left block_steps => block_steps
          | Right (err,ps) =>
            let
              val chain = join " -> " (List.map IPoint.toString (point :: ps))
            in
              raise Error ("chasing blockLedges pointers found " ^ err ^
                           " point: " ^ chain)
            end

      fun checkNeighbours block blockSide (pt,ledges) =
          let
            val point = IPoint.fromPoint pt
          in
            if isEmpty point then ledges + 1
            else
              let
                val (block',_) = getRoot point
                val blockSide' = findSide sideStones block'
                val _ =
                    if blockSide' = blockSide then
                      block' = block orelse
                      raise Error
                        ("neighbouring same-side stone is in a " ^
                         "different block " ^ toStringBlock block')
                    else
                      block' <> block orelse
                      raise Error ("neighbouring different-side stones are " ^
                                   "in the same block")
              in
                ledges
              end
          end
          handle Error err =>
            raise Error
              ("checkNeighbours:\n" ^
               "  block = " ^ toStringBlock block ^ "\n" ^
               "  blockSide = " ^ Side.toString blockSide ^ "\n" ^
               "  pt = " ^ Point.toString pt ^ "\n" ^ err)

      fun checkPoints (point,acc) =
          (if isEdge point then acc
           else
             let
               val (emptySet,totalStepsToRoot,blockMap) = acc

               val pt as Point.Point {file,rank} = IPoint.toPoint point
             in
               if isEmpty point then
                 (PointSet.add emptySet pt, totalStepsToRoot, blockMap)
               else
                 let
                   val (block,totalStepsToRoot) = getRoot point
                   val blockPt = IPoint.toPoint block
                   val blockSide = findSide sideStones block
                   val (blockStones,blockLedges) =
                       Option.getOpt (PointMap.peek blockMap blockPt, (0,0))
                   val blockStones = blockStones + 1
                   val neighbourPts =
                       Dimensions.neighbours IPoint.DIMENSIONS pt
                   val blockLedges =
                       PointSet.foldl (checkNeighbours block blockSide)
                         blockLedges neighbourPts
                   val blockMap =
                       PointMap.insert blockMap
                         (blockPt,(blockStones,blockLedges))
                 in
                   (emptySet,totalStepsToRoot,blockMap)
                 end
             end)
          handle Error err =>
            raise Error
              ("checkPoints: point = " ^ IPoint.toString point ^ "\n" ^ err)

      fun checkBlocks (pt,(numStones,numLedges)) =
          let
            val point = IPoint.fromPoint pt
            val numStones' = findStones sideStones point
            val numLedges' = findLedges blockLedges point
            val _ = numStones = numStones' orelse
                    (Print.trace Print.ppInt "stored stones" numStones';
                     Print.trace Print.ppInt "calculated stones" numStones;
                     raise Error "wrong number of stones in block")
            val _ = numLedges = numLedges' orelse
                    (Print.trace Print.ppInt "stored ledges" numLedges';
                     Print.trace Print.ppInt "calculated ledges" numLedges;
                     raise Error "wrong number of ledges in block")
          in
            ()
          end
          handle Error err =>
            raise Error ("checkBlocks: point = "^Point.toString pt^"\n"^err)

      val emptySet = PointSet.empty
      val blockMap : (int * int) PointMap.map = PointMap.new ()
      val totalStepsToRoot = 0
      val acc = (emptySet,totalStepsToRoot,blockMap)
      val acc = List.foldl checkPoints acc IPoint.allPoints
      val (emptySet,totalStepsToRoot,blockMap) = acc
      val nonEmptyPoints = length IPoint.boardPoints - PointSet.size emptySet

(*GomiTrace1
      val () =
          if nonEmptyPoints = 0 then ()
          else
            let
              val averageStepsToRoot =
                  Real.fromInt totalStepsToRoot / Real.fromInt nonEmptyPoints
            in
              Print.trace
                Print.ppReal "checkBoard.averageStepsToRoot" averageStepsToRoot
            end
*)

      val () = PointMap.app checkBlocks blockMap

      fun ptsToSet l = PointSet.fromList (List.map IPoint.toPoint l)
      val ppEmpty = IIntSet.pp IPoint.pp
      val _ =
          PointSet.equal emptySet (ptsToSet (IIntSet.toList empty)) orelse
          (Print.trace ppEmpty "stored empty set" empty;
           Print.trace PointSet.pp "calculated empty set" emptySet;
           raise Error "bad empty set")

      val lastMoveCapturedSet = ptsToSet (IStack.toList lastMoveCaptured)
      val ppLastMoveCaptured = IStack.pp IPoint.pp
      val _ =
          not (knownLastMove lastMove) orelse
          PointSet.size lastMoveCapturedSet =
          IStack.size lastMoveCaptured orelse
          let
            val () =
                Print.trace
                  ppLastMoveCaptured "lastMoveCaptured" lastMoveCaptured
          in
            raise Error "duplicates in lastMoveCaptured"
          end
      val _ =
          not (knownLastMove lastMove) orelse
          PointSet.subset lastMoveCapturedSet emptySet orelse
          let
            val () =
                Print.trace
                  ppLastMoveCaptured "lastMoveCaptured" lastMoveCaptured
            val () =
                Print.trace PointSet.pp "calculated empty set" emptySet
          in
            raise Error "lastMoveCaptured point is not empty"
          end

      val _ =
          ko = IPoint.NO_POINT orelse
          isEmptyPoint blockLedges ko orelse
          raise Error "ko point is not empty"
    in
      ()
    end
    handle Error err => raise Bug ("IBoard.check: "^err)
         | Bug bug => raise Bug ("IBoard.check: "^bug);
*)

(* ------------------------------------------------------------------------- *)
(* Mapping to and from purely functional boards.                             *)
(* ------------------------------------------------------------------------- *)

fun toBoard board =
    let
      val dim = IPoint.DIMENSIONS

      fun peekFn pt = peek board (IPoint.fromPoint pt)
    in
      Board.tabulate dim peekFn
    end;

fun fromBoard brd =
    let
(*GomiDebug
      val _ = Board.dimensions brd = IPoint.DIMENSIONS orelse
              raise Bug "IBoard.fromBoard: bad dimensions"
*)

      fun addLedge (pt,ledges) =
          if Board.isEmpty brd pt then ledges + 1 else ledges

      val blockLedges = Array.array (IPoint.POINTS,EDGE_POINT)
      and sideStones = Array.array (IPoint.POINTS,0)
      and empty = IIntSet.empty IPoint.POINTS

      fun processPoint pt =
          let
            val point = IPoint.fromPoint pt
          in
            case Board.peek brd pt of
              SOME (side,blk) =>
              let
                val blkStones = Board.blockStones blk
                val ledges = PointSet.foldl addLedge 0 (Board.neighbours brd pt)
                val block = IPoint.fromPoint (PointSet.pick blkStones)
                val () = incrementLedges blockLedges block ledges
                val () = incrementStonesSide sideStones side block 1
              in
                if block = point then edgeToZeroLedges blockLedges block
                else Array.update (blockLedges,point,block)
              end
            | NONE =>
              let
                val () = Array.update (blockLedges,point,EMPTY_POINT)
                val () = IIntSet.add empty point
              in
                ()
              end
          end

      val ko = ref IPoint.NO_POINT
      and lastMove = newLastMove ()

      val () = PointSet.app processPoint (Board.points brd)

      val board =
          Board
            {blockLedges = blockLedges,
             sideStones = sideStones,
             empty = empty,
             ko = ko,
             lastMove = lastMove}

(*GomiDebug
      val () = check board
*)
    in
      board
    end
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.fromBoard: " ^ bug)
         | e => raise Bug ("IBoard.fromBoard: " ^ exnMessage e);
*)

(* ------------------------------------------------------------------------- *)
(* Blocks.                                                                   *)
(* ------------------------------------------------------------------------- *)

fun block (Board {blockLedges,...}) point =
    let
      val b = peekBlock blockLedges point
(*GomiDebug
      val _ = b <> EDGE_BLOCK orelse raise Bug "edge block"
      val _ = b <> EMPTY_BLOCK orelse raise Bug "empty block"
      val _ = b <> DISAPPEARING_BLOCK orelse raise Bug "disappearing block"
*)
    in
      b
    end
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.block: " ^ bug);
*)

fun stones (Board {sideStones,...}) block =
    findStones sideStones block
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.stones: " ^ bug);
*)

fun ledges (Board {blockLedges,...}) block =
    findLedges blockLedges block
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.ledges: " ^ bug);
*)

(* ------------------------------------------------------------------------- *)
(* Classifying neighbouring points.                                          *)
(* ------------------------------------------------------------------------- *)

datatype neighbour =
    Edge
  | Empty
  | Friend of int
  | Enemy of int;

local
  val allFriends = Vector.tabulate (IPoint.POINTS,Friend)
  and allEnemies = Vector.tabulate (IPoint.POINTS,Enemy);
in
  (* "Constructors" that avoid allocation *)
  fun Friend' b = Vector.sub (allFriends,b);
  fun Enemy' b = Vector.sub (allEnemies,b);
end;

fun identifyNeighbour blockLedges sideStones side point =
    let
      val block = peekBlock blockLedges point
(*GomiDebug
      val _ = block <> DISAPPEARING_BLOCK orelse
              raise Bug "disappearing block"
*)
    in
      if block = EDGE_BLOCK then Edge
      else if block = EMPTY_BLOCK then Empty
      else if Side.equal (findSide sideStones block) side then Friend' block
      else Enemy' block
    end
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.identifyNeighbour: " ^ bug)
         | e => raise Bug ("IBoard.identifyNeighbour: " ^ exnMessage e);
*)

(* ------------------------------------------------------------------------- *)
(* Placing stones that don't fill one-point block eyes.                      *)
(* ------------------------------------------------------------------------- *)

local
  fun tooManyEdges () =
      let
(*GomiDebug
        val () = raise Bug "too many edge neighbours"
*)
      in
        false
      end;

  fun placeSingleStone bl ss s p ledges =
      let
        val () = storeLedges bl p ledges
        val () = storeSideStones ss p s 1
      in
        ()
      end;

  fun class00 bl ss ko lb lc s p fl (*GomiDebug()*) =
      let
        (* no need to check for suicide: the board topology implies *)
        (* that fl must be positive *)
(*GomiDebug
        val _ = fl > 0 orelse raise Bug "class00: nonpositive fl"
*)
        val () = IStack.reset lb
        val () = IStack.reset lc
        val () = ko := IPoint.NO_POINT
        val () = placeSingleStone bl ss s p fl
      in
        true
      end;

  fun class01 bl ss ko lb lc s p fl (*GomiDebug()*) e1 e1c =
      if findLedges bl e1 = e1c then
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = if singleStone ss e1 then ko := e1
                   else ko := IPoint.NO_POINT
          val () = captureBlock bl ss lc e1
          val () = placeSingleStone bl ss s p (fl + e1c)
        in
          true
        end
      else
        fl > 0 andalso
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = decrementLedges bl e1 e1c
          val () = placeSingleStone bl ss s p fl
        in
          true
        end;

  fun class02 bl ss ko lb lc s p fl (*GomiDebug()*) e1 e1c e2 e2c =
      if findLedges bl e1 = e1c then
        if findLedges bl e2 = e2c then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = captureBlock bl ss lc e2
            val () = placeSingleStone bl ss s p (fl + e1c + e2c)
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = if singleStone ss e1 then ko := e1
                     else ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = decrementLedges bl e2 e2c
            val () = placeSingleStone bl ss s p (fl + e1c)
          in
            true
          end
      else if findLedges bl e2 = e2c then
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = if singleStone ss e2 then ko := e2
                   else ko := IPoint.NO_POINT
          val () = decrementLedges bl e1 e1c
          val () = captureBlock bl ss lc e2
          val () = placeSingleStone bl ss s p (fl + e2c)
        in
          true
        end
      else
        fl > 0 andalso
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = decrementLedges bl e1 e1c
          val () = decrementLedges bl e2 e2c
          val () = placeSingleStone bl ss s p fl
        in
          true
        end;

  fun class03 bl ss ko lb lc s p fl (*GomiDebug()*) e1 e1c e2 e3 =
      if findLedges bl e1 = e1c then
        if findLedges bl e2 = 1 then
          if findLedges bl e3 = 1 then
            let
              val () = IStack.reset lb
              val () = IStack.reset lc
              val () = ko := IPoint.NO_POINT
              val () = captureBlock bl ss lc e1
              val () = captureBlock bl ss lc e2
              val () = captureBlock bl ss lc e3
              val () = placeSingleStone bl ss s p (fl + e1c + 2)
            in
              true
            end
          else
            let
              val () = IStack.reset lb
              val () = IStack.reset lc
              val () = ko := IPoint.NO_POINT
              val () = captureBlock bl ss lc e1
              val () = captureBlock bl ss lc e2
              val () = decrementLedges bl e3 1
              val () = placeSingleStone bl ss s p (fl + e1c + 1)
            in
              true
            end
        else if findLedges bl e3 = 1 then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = decrementLedges bl e2 1
            val () = captureBlock bl ss lc e3
            val () = placeSingleStone bl ss s p (fl + e1c + 1)
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = if singleStone ss e1 then ko := e1
                     else ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = decrementLedges bl e2 1
            val () = decrementLedges bl e3 1
            val () = placeSingleStone bl ss s p (fl + e1c)
          in
            true
          end
      else if findLedges bl e2 = 1 then
        if findLedges bl e3 = 1 then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 e1c
            val () = captureBlock bl ss lc e2
            val () = captureBlock bl ss lc e3
            val () = placeSingleStone bl ss s p (fl + 2)
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = if singleStone ss e2 then ko := e2
                     else ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 e1c
            val () = captureBlock bl ss lc e2
            val () = decrementLedges bl e3 1
            val () = placeSingleStone bl ss s p (fl + 1)
          in
            true
          end
      else if findLedges bl e3 = 1 then
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = if singleStone ss e3 then ko := e3
                   else ko := IPoint.NO_POINT
          val () = decrementLedges bl e1 e1c
          val () = decrementLedges bl e2 1
          val () = captureBlock bl ss lc e3
          val () = placeSingleStone bl ss s p (fl + 1)
        in
          true
        end
      else
        fl > 0 andalso
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = decrementLedges bl e1 e1c
          val () = decrementLedges bl e2 1
          val () = decrementLedges bl e3 1
          val () = placeSingleStone bl ss s p fl
        in
          true
        end;

  fun class04 bl ss ko lb lc s p (*GomiDebug()*) e1 e2 e3 e4 =
      if findLedges bl e1 = 1 then
        if findLedges bl e2 = 1 then
          if findLedges bl e3 = 1 then
            if findLedges bl e4 = 1 then
              let
                val () = IStack.reset lb
                val () = IStack.reset lc
                val () = ko := IPoint.NO_POINT
                val () = captureBlock bl ss lc e1
                val () = captureBlock bl ss lc e2
                val () = captureBlock bl ss lc e3
                val () = captureBlock bl ss lc e4
                val () = placeSingleStone bl ss s p 4
              in
                true
              end
            else
              let
                val () = IStack.reset lb
                val () = IStack.reset lc
                val () = ko := IPoint.NO_POINT
                val () = captureBlock bl ss lc e1
                val () = captureBlock bl ss lc e2
                val () = captureBlock bl ss lc e3
                val () = decrementLedges bl e4 1
                val () = placeSingleStone bl ss s p 3
              in
               true
              end
          else if findLedges bl e4 = 1 then
              let
                val () = IStack.reset lb
                val () = IStack.reset lc
                val () = ko := IPoint.NO_POINT
                val () = captureBlock bl ss lc e1
                val () = captureBlock bl ss lc e2
                val () = decrementLedges bl e3 1
                val () = captureBlock bl ss lc e4
                val () = placeSingleStone bl ss s p 3
              in
                true
              end
          else
            let
              val () = IStack.reset lb
              val () = IStack.reset lc
              val () = ko := IPoint.NO_POINT
              val () = captureBlock bl ss lc e1
              val () = captureBlock bl ss lc e2
              val () = decrementLedges bl e3 1
              val () = decrementLedges bl e4 1
              val () = placeSingleStone bl ss s p 2
            in
              true
            end
        else if findLedges bl e3 = 1 then
          if findLedges bl e4 = 1 then
            let
              val () = IStack.reset lb
              val () = IStack.reset lc
              val () = ko := IPoint.NO_POINT
              val () = captureBlock bl ss lc e1
              val () = decrementLedges bl e2 1
              val () = captureBlock bl ss lc e3
              val () = captureBlock bl ss lc e4
              val () = placeSingleStone bl ss s p 3
            in
              true
            end
          else
            let
              val () = IStack.reset lb
              val () = IStack.reset lc
              val () = ko := IPoint.NO_POINT
              val () = captureBlock bl ss lc e1
              val () = decrementLedges bl e2 1
              val () = captureBlock bl ss lc e3
              val () = decrementLedges bl e4 1
              val () = placeSingleStone bl ss s p 2
            in
              true
            end
        else if findLedges bl e4 = 1 then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = decrementLedges bl e2 1
            val () = decrementLedges bl e3 1
            val () = captureBlock bl ss lc e4
            val () = placeSingleStone bl ss s p 2
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = if singleStone ss e1 then ko := e1
                     else ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = decrementLedges bl e2 1
            val () = decrementLedges bl e3 1
            val () = decrementLedges bl e4 1
            val () = placeSingleStone bl ss s p 1
          in
            true
          end
      else if findLedges bl e2 = 1 then
        if findLedges bl e3 = 1 then
          if findLedges bl e4 = 1 then
            let
              val () = IStack.reset lb
              val () = IStack.reset lc
              val () = ko := IPoint.NO_POINT
              val () = decrementLedges bl e1 1
              val () = captureBlock bl ss lc e2
              val () = captureBlock bl ss lc e3
              val () = captureBlock bl ss lc e4
              val () = placeSingleStone bl ss s p 3
            in
              true
            end
          else
            let
              val () = IStack.reset lb
              val () = IStack.reset lc
              val () = ko := IPoint.NO_POINT
              val () = decrementLedges bl e1 1
              val () = captureBlock bl ss lc e2
              val () = captureBlock bl ss lc e3
              val () = decrementLedges bl e4 1
              val () = placeSingleStone bl ss s p 2
            in
              true
            end
        else if findLedges bl e4 = 1 then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 1
            val () = captureBlock bl ss lc e2
            val () = decrementLedges bl e3 1
            val () = captureBlock bl ss lc e4
            val () = placeSingleStone bl ss s p 2
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = if singleStone ss e2 then ko := e2
                     else ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 1
            val () = captureBlock bl ss lc e2
            val () = decrementLedges bl e3 1
            val () = decrementLedges bl e4 1
            val () = placeSingleStone bl ss s p 1
          in
            true
          end
      else if findLedges bl e3 = 1 then
        if findLedges bl e4 = 1 then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 1
            val () = decrementLedges bl e2 1
            val () = captureBlock bl ss lc e3
            val () = captureBlock bl ss lc e4
            val () = placeSingleStone bl ss s p 2
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = if singleStone ss e3 then ko := e3
                     else ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 1
            val () = decrementLedges bl e2 1
            val () = captureBlock bl ss lc e3
            val () = decrementLedges bl e4 1
            val () = placeSingleStone bl ss s p 1
          in
            true
          end
      else
        findLedges bl e4 = 1 andalso
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = if singleStone ss e4 then ko := e4
                   else ko := IPoint.NO_POINT
          val () = decrementLedges bl e1 1
          val () = decrementLedges bl e2 1
          val () = decrementLedges bl e3 1
          val () = captureBlock bl ss lc e4
          val () = placeSingleStone bl ss s p 1
        in
          true
        end;

  fun class10 bl ss ko lb lc s p f1 fl (*GomiDebug()*) =
      let
        val fl1 = findLedges bl f1
      in
        fl1 + fl > 0 andalso
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = extendBlock bl ss p f1 fl
        in
          true
        end
      end;

  fun class11 bl ss ko lb lc s p f1 fl (*GomiDebug()*) e1 e1c =
      if findLedges bl e1 = e1c then
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = captureBlock bl ss lc e1
          val () = extendBlock bl ss p f1 (fl + e1c)
        in
          true
        end
      else
        let
          val fl1 = findLedges bl f1
        in
          fl1 + fl > 0 andalso
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 e1c
            val () = extendBlock bl ss p f1 fl
          in
            true
          end
        end;

  fun class12 bl ss ko lb lc s p f1 fl (*GomiDebug()*) e1 e1c e2 =
      if findLedges bl e1 = e1c then
        if findLedges bl e2 = 1 then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = captureBlock bl ss lc e2
            val () = extendBlock bl ss p f1 (fl + e1c + 1)
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = decrementLedges bl e2 1
            val () = extendBlock bl ss p f1 (fl + e1c)
          in
            true
          end
      else if findLedges bl e2 = 1 then
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = decrementLedges bl e1 e1c
          val () = captureBlock bl ss lc e2
          val () = extendBlock bl ss p f1 (fl + 1)
        in
          true
        end
      else
        let
          val fl1 = findLedges bl f1
        in
          fl1 + fl > 0 andalso
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 e1c
            val () = decrementLedges bl e2 1
            val () = extendBlock bl ss p f1 fl
          in
            true
          end
        end;

  fun class13 bl ss ko lb lc s p f1 (*GomiDebug()*) e1 e2 e3 =
      if findLedges bl e1 = 1 then
        if findLedges bl e2 = 1 then
          if findLedges bl e3 = 1 then
            let
              val () = IStack.reset lb
              val () = IStack.reset lc
              val () = ko := IPoint.NO_POINT
              val () = captureBlock bl ss lc e1
              val () = captureBlock bl ss lc e2
              val () = captureBlock bl ss lc e3
              val () = extendBlock bl ss p f1 2
            in
              true
            end
          else
            let
              val () = IStack.reset lb
              val () = IStack.reset lc
              val () = ko := IPoint.NO_POINT
              val () = captureBlock bl ss lc e1
              val () = captureBlock bl ss lc e2
              val () = decrementLedges bl e3 1
              val () = extendBlock bl ss p f1 1
            in
              true
            end
        else if findLedges bl e3 = 1 then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = decrementLedges bl e2 1
            val () = captureBlock bl ss lc e3
            val () = extendBlock bl ss p f1 1
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = decrementLedges bl e2 1
            val () = decrementLedges bl e3 1
            val () = extendBlock bl ss p f1 0
          in
            true
          end
      else if findLedges bl e2 = 1 then
        if findLedges bl e3 = 1 then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 1
            val () = captureBlock bl ss lc e2
            val () = captureBlock bl ss lc e3
            val () = extendBlock bl ss p f1 1
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 1
            val () = captureBlock bl ss lc e2
            val () = decrementLedges bl e3 1
            val () = extendBlock bl ss p f1 0
          in
            true
          end
      else if findLedges bl e3 = 1 then
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = decrementLedges bl e1 1
          val () = decrementLedges bl e2 1
          val () = captureBlock bl ss lc e3
          val () = extendBlock bl ss p f1 0
        in
          true
        end
      else
        let
          val fl1 = findLedges bl f1
        in
          fl1 > 1 andalso
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 1
            val () = decrementLedges bl e2 1
            val () = decrementLedges bl e3 1
            val () = extendBlock bl ss p f1 ~1
          in
            true
          end
        end;

  fun class20 bl ss ko lb lc s p f1 f2 fl (*GomiDebug()*) =
      let
        val fl1 = findLedges bl f1
        and fl2 = findLedges bl f2
      in
        fl1 + fl2 + fl > 0 andalso
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = connectTwoBlocks bl ss lb p f1 f2 fl
        in
          true
        end
      end;

  fun class21 bl ss ko lb lc s p f1 f2 fl (*GomiDebug()*) e1 e1c =
      if findLedges bl e1 = e1c then
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = captureBlock bl ss lc e1
          val () = connectTwoBlocks bl ss lb p f1 f2 (fl + e1c)
        in
          true
        end
      else
        let
          val fl1 = findLedges bl f1
          and fl2 = findLedges bl f2
        in
          fl1 + fl2 + fl > 0 andalso
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 e1c
            val () = connectTwoBlocks bl ss lb p f1 f2 fl
          in
            true
          end
        end;

  fun class22 bl ss ko lb lc s p f1 f2 (*GomiDebug()*) e1 e2 =
      if findLedges bl e1 = 1 then
        if findLedges bl e2 = 1 then
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = captureBlock bl ss lc e2
            val () = connectTwoBlocks bl ss lb p f1 f2 0
          in
            true
          end
        else
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = captureBlock bl ss lc e1
            val () = decrementLedges bl e2 1
            val () = connectTwoBlocks bl ss lb p f1 f2 ~1
          in
            true
          end
      else if findLedges bl e2 = 1 then
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = decrementLedges bl e1 1
          val () = captureBlock bl ss lc e2
          val () = connectTwoBlocks bl ss lb p f1 f2 ~1
        in
          true
        end
      else
        let
          val fl1 = findLedges bl f1
          and fl2 = findLedges bl f2
        in
          fl1 + fl2 > 2 andalso
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 1
            val () = decrementLedges bl e2 1
            val () = connectTwoBlocks bl ss lb p f1 f2 ~2
          in
            true
          end
        end;

  fun class30 bl ss ko lb lc s p f1 f2 f3 fl (*GomiDebug()*) =
      let
        val fl1 = findLedges bl f1
        and fl2 = findLedges bl f2
        and fl3 = findLedges bl f3
      in
        fl1 + fl2 + fl3 + fl > 0 andalso
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = connectThreeBlocks bl ss lb p f1 f2 f3 fl
        in
          true
        end
      end;

  fun class31 bl ss ko lb lc s p f1 f2 f3 (*GomiDebug()*) e1 =
      if findLedges bl e1 = 1 then
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = captureBlock bl ss lc e1
          val () = connectThreeBlocks bl ss lb p f1 f2 f3 ~2
        in
          true
        end
      else
        let
          val fl1 = findLedges bl f1
          and fl2 = findLedges bl f2
          and fl3 = findLedges bl f3
        in
          fl1 + fl2 + fl3 > 3 andalso
          let
            val () = IStack.reset lb
            val () = IStack.reset lc
            val () = ko := IPoint.NO_POINT
            val () = decrementLedges bl e1 1
            val () = connectThreeBlocks bl ss lb p f1 f2 f3 ~3
          in
            true
          end
        end;

  fun class40 bl ss ko lb lc s p f1 f2 f3 f4 (*GomiDebug()*) =
      let
        val fl1 = findLedges bl f1
        and fl2 = findLedges bl f2
        and fl3 = findLedges bl f3
        and fl4 = findLedges bl f4
      in
        fl1 + fl2 + fl3 + fl4 > 4 andalso
        let
          val () = IStack.reset lb
          val () = IStack.reset lc
          val () = ko := IPoint.NO_POINT
          val () = connectFourBlocks bl ss lb p f1 f2 f3 f4
        in
          true
        end
      end;

  fun group00 bl ss ko lb lc s p fl (*GomiDebug()*) =
      class00 bl ss ko lb lc s p fl (*GomiDebug()*);

  fun group01 bl ss ko lb lc s p fl (*GomiDebug()*) e1 =
      class01 bl ss ko lb lc s p fl (*GomiDebug()*) e1 1;

  fun group02 bl ss ko lb lc s p fl (*GomiDebug()*) e1 e2 =
      if e1 = e2 then
        class01 bl ss ko lb lc s p fl (*GomiDebug()*) e1 2
      else
        class02 bl ss ko lb lc s p fl (*GomiDebug()*) e1 1 e2 1;

  fun group03 bl ss ko lb lc s p fl (*GomiDebug()*) e1 e2 e3 =
      if e1 = e2 then
        if e1 = e3 then
          class01 bl ss ko lb lc s p fl (*GomiDebug()*) e1 3
        else
          class02 bl ss ko lb lc s p fl (*GomiDebug()*) e1 2 e3 1
      else if e1 = e3 then
        class02 bl ss ko lb lc s p fl (*GomiDebug()*) e1 2 e2 1
      else if e2 = e3 then
        class02 bl ss ko lb lc s p fl (*GomiDebug()*) e2 2 e1 1
      else
        class03 bl ss ko lb lc s p fl (*GomiDebug()*) e1 1 e2 e3;

  fun group04 bl ss ko lb lc s p (*GomiDebug()*) e1 e2 e3 e4 =
      if e1 = e2 then
        if e1 = e3 then
          if e1 = e4 then
            class01 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 4
          else
            class02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 3 e4 1
        else if e1 = e4 then
          class02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 3 e3 1
        else if e3 = e4 then
          class02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 2 e3 2
        else
          class03 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 2 e3 e4
      else if e1 = e3 then
        if e1 = e4 then
          class02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 3 e2 1
        else if e2 = e4 then
          class02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 2 e2 2
        else
          class03 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 2 e2 e4
      else if e2 = e3 then
        if e1 = e4 then
          class02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 2 e2 2
        else if e2 = e4 then
          class02 bl ss ko lb lc s p 0 (*GomiDebug()*) e2 3 e1 1
        else
          class03 bl ss ko lb lc s p 0 (*GomiDebug()*) e2 2 e1 e4
      else if e1 = e4 then
        class03 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 2 e2 e3
      else if e2 = e4 then
        class03 bl ss ko lb lc s p 0 (*GomiDebug()*) e2 2 e1 e3
      else if e3 = e4 then
        class03 bl ss ko lb lc s p 0 (*GomiDebug()*) e3 2 e1 e2
      else
        class04 bl ss ko lb lc s p (*GomiDebug()*) e1 e2 e3 e4;

  fun group10 bl ss ko lb lc s p f1 fl (*GomiDebug()*) =
      class10 bl ss ko lb lc s p f1 fl (*GomiDebug()*);

  fun group11 bl ss ko lb lc s p f1 fl (*GomiDebug()*) e1 =
      class11 bl ss ko lb lc s p f1 fl (*GomiDebug()*) e1 1;

  fun group12 bl ss ko lb lc s p f1 fl (*GomiDebug()*) e1 e2 =
      if e1 = e2 then
        class11 bl ss ko lb lc s p f1 fl (*GomiDebug()*) e1 2
      else
        class12 bl ss ko lb lc s p f1 fl (*GomiDebug()*) e1 1 e2;

  fun group13 bl ss ko lb lc s p f1 (*GomiDebug()*) e1 e2 e3 =
      if e1 = e2 then
        if e1 = e3 then
          class11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 3
        else
          class12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 2 e3
      else if e1 = e3 then
        class12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 2 e2
      else if e2 = e3 then
        class12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e2 2 e1
      else
        class13 bl ss ko lb lc s p f1 (*GomiDebug()*) e1 e2 e3;

  fun group20 bl ss ko lb lc s p f1 f2 fl (*GomiDebug()*) =
      if f1 = f2 then
        (* can fill corner block eyes *)
        class10 bl ss ko lb lc s p f1 fl (*GomiDebug()*)
      else
        class20 bl ss ko lb lc s p f1 f2 fl (*GomiDebug()*);

  fun group21 bl ss ko lb lc s p f1 f2 fl (*GomiDebug()*) e1 =
      if f1 = f2 then
        class11 bl ss ko lb lc s p f1 fl (*GomiDebug()*) e1 1
      else
        class21 bl ss ko lb lc s p f1 f2 fl (*GomiDebug()*) e1 1;

  fun group22 bl ss ko lb lc s p f1 f2 (*GomiDebug()*) e1 e2 =
      if f1 = f2 then
        if e1 = e2 then
          class11 bl ss ko lb lc s p f1 ~2 (*GomiDebug()*) e1 2
        else
          class12 bl ss ko lb lc s p f1 ~2 (*GomiDebug()*) e1 1 e2
      else
        if e1 = e2 then
          class21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1 2
        else
          class22 bl ss ko lb lc s p f1 f2 (*GomiDebug()*) e1 e2;

  fun group30 bl ss ko lb lc s p f1 f2 f3 fl (*GomiDebug()*) =
      if f1 = f2 then
        if f1 = f3 then
          fl <> ~3 andalso  (* must not fill edge block eye *)
          class10 bl ss ko lb lc s p f1 fl (*GomiDebug()*)
        else
          class20 bl ss ko lb lc s p f1 f3 fl (*GomiDebug()*)
      else if f1 = f3 orelse f2 = f3 then
        class20 bl ss ko lb lc s p f1 f2 fl (*GomiDebug()*)
      else
        class30 bl ss ko lb lc s p f1 f2 f3 fl (*GomiDebug()*);

  fun group31 bl ss ko lb lc s p f1 f2 f3 (*GomiDebug()*) e1 =
      if f1 = f2 then
        if f1 = f3 then
          class11 bl ss ko lb lc s p f1 ~3 (*GomiDebug()*) e1 1
        else
          class21 bl ss ko lb lc s p f1 f3 ~3 (*GomiDebug()*) e1 1
      else if f1 = f3 orelse f2 = f3 then
        class21 bl ss ko lb lc s p f1 f2 ~3 (*GomiDebug()*) e1 1
      else
        class31 bl ss ko lb lc s p f1 f2 f3 (*GomiDebug()*) e1;

  fun group40 bl ss ko lb lc s p f1 f2 f3 f4 (*GomiDebug()*) =
      if f1 = f2 then
        if f1 = f3 then
          f1 <> f4 andalso  (* must not fill middle block eye *)
          class20 bl ss ko lb lc s p f1 f4 ~4 (*GomiDebug()*)
        else if f1 = f4 orelse f3 = f4 then
          class20 bl ss ko lb lc s p f1 f3 ~4 (*GomiDebug()*)
        else
          class30 bl ss ko lb lc s p f1 f3 f4 ~4 (*GomiDebug()*)
      else if f1 = f3 orelse f2 = f3 then
        if f1 = f4 orelse f2 = f4 then
          class20 bl ss ko lb lc s p f1 f2 ~4 (*GomiDebug()*)
        else
          class30 bl ss ko lb lc s p f1 f2 f4 ~4 (*GomiDebug()*)
      else if f1 = f4 orelse f2 = f4 orelse f3 = f4 then
        class30 bl ss ko lb lc s p f1 f2 f3 ~4 (*GomiDebug()*)
      else
        class40 bl ss ko lb lc s p f1 f2 f3 f4 (*GomiDebug()*);
in
  fun playSensibleStoneMove board s p =
      let
        val Board
              {blockLedges = bl,
               sideStones = ss,
               empty,
               ko as ref k,
               lastMove} = board

        val LastMove
              {status = lastMoveStatus,
               point = lastMovePoint,
               blocks = lb,
               captured = lc,
               ko = lastMoveKo} = lastMove

(*GomiDebug
        val _ = peekBlock bl p = EMPTY_BLOCK orelse
                raise Bug "point not empty"
*)
      in
        p <> k andalso
        let
          val n1 = identifyNeighbour bl ss s (IPoint.moveUp p)
          and n2 = identifyNeighbour bl ss s (IPoint.moveLeft p)
          and n3 = identifyNeighbour bl ss s (IPoint.moveRight p)
          and n4 = identifyNeighbour bl ss s (IPoint.moveDown p)
        in
          (* Automatically generated by scripts/bigcase *)
          case n1 of
            Edge =>
            (case n2 of
               Edge =>
               (case n3 of
                  Edge => tooManyEdges ()
                | Empty =>
                  (case n4 of
                     Edge => tooManyEdges ()
                   | Empty =>
                     group00 bl ss ko lb lc s p 2 (*GomiDebug()*)
                   | Friend f1 =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Enemy e1 =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1)
                | Friend f1 =>
                  (case n4 of
                     Edge => tooManyEdges ()
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1)
                | Enemy e1 =>
                  (case n4 of
                     Edge => tooManyEdges ()
                   | Empty =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2))
             | Empty =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge => tooManyEdges ()
                   | Empty =>
                     group00 bl ss ko lb lc s p 2 (*GomiDebug()*)
                   | Friend f1 =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Enemy e1 =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group00 bl ss ko lb lc s p 2 (*GomiDebug()*)
                   | Empty =>
                     group00 bl ss ko lb lc s p 3 (*GomiDebug()*)
                   | Friend f1 =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Enemy e1 =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1)
                | Friend f1 =>
                  (case n4 of
                     Edge =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1)
                | Enemy e1 =>
                  (case n4 of
                     Edge =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1
                   | Empty =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2))
             | Friend f1 =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge => tooManyEdges ()
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1)
                | Friend f2 =>
                  (case n4 of
                     Edge =>
                     group20 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*)
                   | Empty =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Friend f3 =>
                     group30 bl ss ko lb lc s p f1 f2 f3 ~3 (*GomiDebug()*)
                   | Enemy e1 =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1)
                | Enemy e1 =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2))
             | Enemy e1 =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge => tooManyEdges ()
                   | Empty =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1
                   | Empty =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2)
                | Friend f1 =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2)
                | Enemy e2 =>
                  (case n4 of
                     Edge =>
                     group02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2
                   | Empty =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2
                   | Friend f1 =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2
                   | Enemy e3 =>
                     group03 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2 e3)))
          | Empty =>
            (case n2 of
               Edge =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge => tooManyEdges ()
                   | Empty =>
                     group00 bl ss ko lb lc s p 2 (*GomiDebug()*)
                   | Friend f1 =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Enemy e1 =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group00 bl ss ko lb lc s p 2 (*GomiDebug()*)
                   | Empty =>
                     group00 bl ss ko lb lc s p 3 (*GomiDebug()*)
                   | Friend f1 =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Enemy e1 =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1)
                | Friend f1 =>
                  (case n4 of
                     Edge =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1)
                | Enemy e1 =>
                  (case n4 of
                     Edge =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1
                   | Empty =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2))
             | Empty =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge =>
                     group00 bl ss ko lb lc s p 2 (*GomiDebug()*)
                   | Empty =>
                     group00 bl ss ko lb lc s p 3 (*GomiDebug()*)
                   | Friend f1 =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Enemy e1 =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group00 bl ss ko lb lc s p 3 (*GomiDebug()*)
                   | Empty =>
                     group00 bl ss ko lb lc s p 4 (*GomiDebug()*)
                   | Friend f1 =>
                     group10 bl ss ko lb lc s p f1 2 (*GomiDebug()*)
                   | Enemy e1 =>
                     group01 bl ss ko lb lc s p 3 (*GomiDebug()*) e1)
                | Friend f1 =>
                  (case n4 of
                     Edge =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 2 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 0 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1)
                | Enemy e1 =>
                  (case n4 of
                     Edge =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1
                   | Empty =>
                     group01 bl ss ko lb lc s p 3 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 2 (*GomiDebug()*) e1 e2))
             | Friend f1 =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 2 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 0 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1)
                | Friend f2 =>
                  (case n4 of
                     Edge =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Empty =>
                     group20 bl ss ko lb lc s p f1 f2 0 (*GomiDebug()*)
                   | Friend f3 =>
                     group30 bl ss ko lb lc s p f1 f2 f3 ~2 (*GomiDebug()*)
                   | Enemy e1 =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1)
                | Enemy e1 =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2))
             | Enemy e1 =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1
                   | Empty =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1
                   | Empty =>
                     group01 bl ss ko lb lc s p 3 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 2 (*GomiDebug()*) e1 e2)
                | Friend f1 =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2)
                | Enemy e2 =>
                  (case n4 of
                     Edge =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2
                   | Empty =>
                     group02 bl ss ko lb lc s p 2 (*GomiDebug()*) e1 e2
                   | Friend f1 =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2
                   | Enemy e3 =>
                     group03 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2 e3)))
          | Friend f1 =>
            (case n2 of
               Edge =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge => tooManyEdges ()
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1)
                | Friend f2 =>
                  (case n4 of
                     Edge =>
                     group20 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*)
                   | Empty =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Friend f3 =>
                     group30 bl ss ko lb lc s p f1 f2 f3 ~3 (*GomiDebug()*)
                   | Enemy e1 =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1)
                | Enemy e1 =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2))
             | Empty =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge =>
                     group10 bl ss ko lb lc s p f1 0 (*GomiDebug()*)
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group10 bl ss ko lb lc s p f1 1 (*GomiDebug()*)
                   | Empty =>
                     group10 bl ss ko lb lc s p f1 2 (*GomiDebug()*)
                   | Friend f2 =>
                     group20 bl ss ko lb lc s p f1 f2 0 (*GomiDebug()*)
                   | Enemy e1 =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1)
                | Friend f2 =>
                  (case n4 of
                     Edge =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Empty =>
                     group20 bl ss ko lb lc s p f1 f2 0 (*GomiDebug()*)
                   | Friend f3 =>
                     group30 bl ss ko lb lc s p f1 f2 f3 ~2 (*GomiDebug()*)
                   | Enemy e1 =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1)
                | Enemy e1 =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2))
             | Friend f2 =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge =>
                     group20 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*)
                   | Empty =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Friend f3 =>
                     group30 bl ss ko lb lc s p f1 f2 f3 ~3 (*GomiDebug()*)
                   | Enemy e1 =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group20 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*)
                   | Empty =>
                     group20 bl ss ko lb lc s p f1 f2 0 (*GomiDebug()*)
                   | Friend f3 =>
                     group30 bl ss ko lb lc s p f1 f2 f3 ~2 (*GomiDebug()*)
                   | Enemy e1 =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1)
                | Friend f3 =>
                  (case n4 of
                     Edge =>
                     group30 bl ss ko lb lc s p f1 f2 f3 ~3 (*GomiDebug()*)
                   | Empty =>
                     group30 bl ss ko lb lc s p f1 f2 f3 ~2 (*GomiDebug()*)
                   | Friend f4 =>
                     group40 bl ss ko lb lc s p f1 f2 f3 f4 (*GomiDebug()*)
                   | Enemy e1 =>
                     group31 bl ss ko lb lc s p f1 f2 f3 (*GomiDebug()*) e1)
                | Enemy e1 =>
                  (case n4 of
                     Edge =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1
                   | Empty =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1
                   | Friend f3 =>
                     group31 bl ss ko lb lc s p f1 f2 f3 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group22 bl ss ko lb lc s p f1 f2 (*GomiDebug()*) e1 e2))
             | Enemy e1 =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2)
                | Friend f2 =>
                  (case n4 of
                     Edge =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1
                   | Empty =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1
                   | Friend f3 =>
                     group31 bl ss ko lb lc s p f1 f2 f3 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group22 bl ss ko lb lc s p f1 f2 (*GomiDebug()*) e1 e2)
                | Enemy e2 =>
                  (case n4 of
                     Edge =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2
                   | Empty =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2
                   | Friend f2 =>
                     group22 bl ss ko lb lc s p f1 f2 (*GomiDebug()*) e1 e2
                   | Enemy e3 =>
                     group13 bl ss ko lb lc s p f1 (*GomiDebug()*) e1 e2 e3)))
          | Enemy e1 =>
            (case n2 of
               Edge =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge => tooManyEdges ()
                   | Empty =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1
                   | Empty =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2)
                | Friend f1 =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2)
                | Enemy e2 =>
                  (case n4 of
                     Edge =>
                     group02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2
                   | Empty =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2
                   | Friend f1 =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2
                   | Enemy e3 =>
                     group03 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2 e3))
             | Empty =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge =>
                     group01 bl ss ko lb lc s p 1 (*GomiDebug()*) e1
                   | Empty =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group01 bl ss ko lb lc s p 2 (*GomiDebug()*) e1
                   | Empty =>
                     group01 bl ss ko lb lc s p 3 (*GomiDebug()*) e1
                   | Friend f1 =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group02 bl ss ko lb lc s p 2 (*GomiDebug()*) e1 e2)
                | Friend f1 =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2)
                | Enemy e2 =>
                  (case n4 of
                     Edge =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2
                   | Empty =>
                     group02 bl ss ko lb lc s p 2 (*GomiDebug()*) e1 e2
                   | Friend f1 =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2
                   | Enemy e3 =>
                     group03 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2 e3))
             | Friend f1 =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group11 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1
                   | Empty =>
                     group11 bl ss ko lb lc s p f1 1 (*GomiDebug()*) e1
                   | Friend f2 =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2)
                | Friend f2 =>
                  (case n4 of
                     Edge =>
                     group21 bl ss ko lb lc s p f1 f2 ~2 (*GomiDebug()*) e1
                   | Empty =>
                     group21 bl ss ko lb lc s p f1 f2 ~1 (*GomiDebug()*) e1
                   | Friend f3 =>
                     group31 bl ss ko lb lc s p f1 f2 f3 (*GomiDebug()*) e1
                   | Enemy e2 =>
                     group22 bl ss ko lb lc s p f1 f2 (*GomiDebug()*) e1 e2)
                | Enemy e2 =>
                  (case n4 of
                     Edge =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2
                   | Empty =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2
                   | Friend f2 =>
                     group22 bl ss ko lb lc s p f1 f2 (*GomiDebug()*) e1 e2
                   | Enemy e3 =>
                     group13 bl ss ko lb lc s p f1 (*GomiDebug()*) e1 e2 e3))
             | Enemy e2 =>
               (case n3 of
                  Edge =>
                  (case n4 of
                     Edge =>
                     group02 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2
                   | Empty =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2
                   | Friend f1 =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2
                   | Enemy e3 =>
                     group03 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2 e3)
                | Empty =>
                  (case n4 of
                     Edge =>
                     group02 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2
                   | Empty =>
                     group02 bl ss ko lb lc s p 2 (*GomiDebug()*) e1 e2
                   | Friend f1 =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2
                   | Enemy e3 =>
                     group03 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2 e3)
                | Friend f1 =>
                  (case n4 of
                     Edge =>
                     group12 bl ss ko lb lc s p f1 ~1 (*GomiDebug()*) e1 e2
                   | Empty =>
                     group12 bl ss ko lb lc s p f1 0 (*GomiDebug()*) e1 e2
                   | Friend f2 =>
                     group22 bl ss ko lb lc s p f1 f2 (*GomiDebug()*) e1 e2
                   | Enemy e3 =>
                     group13 bl ss ko lb lc s p f1 (*GomiDebug()*) e1 e2 e3)
                | Enemy e3 =>
                  (case n4 of
                     Edge =>
                     group03 bl ss ko lb lc s p 0 (*GomiDebug()*) e1 e2 e3
                   | Empty =>
                     group03 bl ss ko lb lc s p 1 (*GomiDebug()*) e1 e2 e3
                   | Friend f1 =>
                     group13 bl ss ko lb lc s p f1 (*GomiDebug()*) e1 e2 e3
                   | Enemy e4 =>
                     group04 bl ss ko lb lc s p (*GomiDebug()*) e1 e2 e3 e4)))
        end andalso
        let
          val () = IIntSet.delete empty p
          val () = IStack.appTopDown (IIntSet.add empty) lc
          val () = lastMoveStatus := StoneLastMove
          val () = lastMovePoint := p
          val () = lastMoveKo := k
(*GomiDebug
          val () =
              if Portable.randomInt PLAY_SENSIBLE_STONE_MOVE_CHECK <> 0 then ()
              else check board
*)
        in
          true
        end
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.playSensibleStoneMove: " ^ bug)
           | e => raise Bug ("IBoard.playSensibleStoneMove: " ^ exnMessage e);
*)
end;

(* ------------------------------------------------------------------------- *)
(* Filling eyes at the end of the game.                                      *)
(* ------------------------------------------------------------------------- *)

local
  fun class1 bl ss lb p b1 side ledges =
      let
        val () = IStack.reset lb
        val () = extendBlock bl ss p b1 ledges
      in
        SOME side
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.class1:\n" ^ bug);
*)

  fun class2 bl ss lb p b1 b2 side ledges =
      let
        val () = IStack.reset lb
        val () = connectTwoBlocks bl ss lb p b1 b2 ledges
      in
        SOME side
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.class2:\n" ^ bug);
*)

  fun class3 bl ss lb p b1 b2 b3 side ledges =
      let
        val () = IStack.reset lb
        val () = connectThreeBlocks bl ss lb p b1 b2 b3 ledges
      in
        SOME side
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.class3:\n" ^ bug);
*)

  fun class4 bl ss lb p b1 b2 b3 b4 side =
      let
        val () = IStack.reset lb
        val () = connectFourBlocks bl ss lb p b1 b2 b3 b4
      in
        SOME side
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.class4:\n" ^ bug);
*)

  fun group11 bl ss lb p d1 side b1 ledges =
      (if b1 = EDGE_BLOCK then
         class1 bl ss lb p d1 side (ledges + 1)
       else if Side.equal side (findSide ss b1) then
         class2 bl ss lb p d1 b1 side ledges
       else
         NONE)
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.group11:\n" ^ bug);
*)

  fun group21 bl ss lb p d1 d2 side b1 ledges =
      (if b1 = EDGE_BLOCK then
         class2 bl ss lb p d1 d2 side (ledges + 1)
       else if Side.equal side (findSide ss b1) then
         class3 bl ss lb p d1 d2 b1 side ledges
       else
         NONE)
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.group21:\n" ^ bug);
*)

  fun group31 bl ss lb p d1 d2 d3 side b1 =
      (if b1 = EDGE_BLOCK then
         class3 bl ss lb p d1 d2 d3 side ~3
       else if Side.equal side (findSide ss b1) then
         class4 bl ss lb p d1 d2 d3 b1 side
       else
         NONE)
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.group31:\n" ^ bug);
*)

  fun group02 bl ss lb p b1 b2 ledges =
      (if b1 = EDGE_BLOCK then
         class1 bl ss lb p b2 (findSide ss b2) (ledges + 1)
       else if b1 = b2 then
         class1 bl ss lb p b2 (findSide ss b2) ledges
       else
         group11 bl ss lb p b1 (findSide ss b1) b2 ledges)
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.group02:\n" ^ bug);
*)

  fun group12 bl ss lb p d1 side b1 b2 ledges =
      (if b1 = EDGE_BLOCK then
         group11 bl ss lb p d1 side b2 (ledges + 1)
       else if b1 = b2 then
         group11 bl ss lb p d1 side b2 ledges
       else if Side.equal side (findSide ss b1) then
         group21 bl ss lb p d1 b1 side b2 ledges
       else
         NONE)
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.group12:\n" ^ bug);
*)

  fun group22 bl ss lb p d1 d2 side b1 b2 =
      (if b1 = EDGE_BLOCK then
         group21 bl ss lb p d1 d2 side b2 ~3
       else if b1 = b2 then
         group21 bl ss lb p d1 d2 side b2 ~4
       else if Side.equal side (findSide ss b1) then
         group31 bl ss lb p d1 d2 b1 side b2
       else
         NONE)
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.group22:\n" ^ bug);
*)

  fun group03 bl ss lb p b1 b2 b3 ledges =
      (if b1 = EDGE_BLOCK then
         group02 bl ss lb p b2 b3 (ledges + 1)
       else if b1 = b2 orelse b1 = b3 then
         group02 bl ss lb p b2 b3 ledges
       else
         group12 bl ss lb p b1 (findSide ss b1) b2 b3 ledges)
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.group03:\n" ^ bug);
*)

  fun group13 bl ss lb p d1 side b1 b2 b3 =
      (if b1 = EDGE_BLOCK then
         group12 bl ss lb p d1 side b2 b3 ~3
       else if b1 = b2 orelse b1 = b3 then
         group12 bl ss lb p d1 side b2 b3 ~4
       else if Side.equal side (findSide ss b1) then
         group22 bl ss lb p d1 b1 side b2 b3
       else
         NONE)
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.group13:\n" ^ bug);
*)

  fun group04 bl ss lb p b1 b2 b3 b4 =
      (if b1 = EDGE_BLOCK then
         group03 bl ss lb p b2 b3 b4 ~3
       else if b1 = b2 orelse b1 = b3 orelse b1 = b4 then
         group03 bl ss lb p b2 b3 b4 ~4
       else
         group13 bl ss lb p b1 (findSide ss b1) b2 b3 b4)
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye.group04:\n" ^ bug);
*)
in
  fun fillEye board point =
      let
        val Board
              {blockLedges,
               sideStones,
               empty,
               ko as ref k,
               lastMove} = board

        val LastMove
              {status = lastMoveStatus,
               point = lastMovePoint,
               blocks = lastMoveBlocks,
               captured = lastMoveCaptured,
               ko = lastMoveKo} = lastMove

(*GomiDebug
        val _ = peekBlock blockLedges point = EMPTY_BLOCK orelse
                raise Bug "point not empty"
*)
        val b1 = peekBlock blockLedges (IPoint.moveUp point)
        and b2 = peekBlock blockLedges (IPoint.moveDown point)
        and b3 = peekBlock blockLedges (IPoint.moveLeft point)
        and b4 = peekBlock blockLedges (IPoint.moveRight point)
(*GomiDebug
        val _ = b1 <> DISAPPEARING_BLOCK orelse
                raise Bug "disappearing block b1"
        val _ = b2 <> DISAPPEARING_BLOCK orelse
                raise Bug "disappearing block b2"
        val _ = b3 <> DISAPPEARING_BLOCK orelse
                raise Bug "disappearing block b3"
        val _ = b4 <> DISAPPEARING_BLOCK orelse
                raise Bug "disappearing block b4"
*)
      in
        if b1 = EMPTY_BLOCK orelse b2 = EMPTY_BLOCK orelse
           b3 = EMPTY_BLOCK orelse b4 = EMPTY_BLOCK then
          NONE
        else
          case group04 blockLedges sideStones lastMoveBlocks point b1 b2 b3 b4 of
            NONE => NONE
          | side as SOME _ =>
            let
              val () = lastMoveStatus := StoneLastMove
              val () = lastMovePoint := point
              val () = IStack.reset lastMoveCaptured
              val () = lastMoveKo := k

              val () = IIntSet.delete empty point
              val () = ko := IPoint.NO_POINT
            in
              side
            end
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.fillEye:\n" ^ bug);
*)
end;

(* ------------------------------------------------------------------------- *)
(* Moves.                                                                    *)
(* ------------------------------------------------------------------------- *)

fun playPassMove board =
    let
      val Board
            {ko as ref k,
             lastMove,
             ...} = board

      val LastMove
            {status = lastMoveStatus,
             captured = lastMoveCaptured,
             ko = lastMoveKo,
             ...} = lastMove

      val () = lastMoveStatus := PassLastMove
      val () = IStack.reset lastMoveCaptured
      val () = lastMoveKo := k

      val () = ko := IPoint.NO_POINT
    in
      ()
    end;

fun playStoneMove board side point =
    playSensibleStoneMove board side point orelse
    let
      val Board
            {blockLedges,
             sideStones,
             ...} = board

      fun checkNeighbour p (block,ledges) =
          let
            val b = peekBlock blockLedges p
(*GomiDebug
            val _ = b <> EMPTY_BLOCK orelse
                    raise Bug "empty block"
            val _ = b <> DISAPPEARING_BLOCK orelse
                    raise Bug "disappearing block"
*)
          in
            if b = EDGE_BLOCK then SOME (block,ledges)
            else if ledges = 0 orelse b = block then SOME (b, ledges + 1)
            else NONE
          end

      val block_ledges = (EMPTY_BLOCK,0)
    in
      case checkNeighbour (IPoint.moveUp point) block_ledges of
        NONE => false
      | SOME block_ledges =>
        case checkNeighbour (IPoint.moveDown point) block_ledges of
          NONE => false
        | SOME block_ledges =>
          case checkNeighbour (IPoint.moveLeft point) block_ledges of
            NONE => false
          | SOME block_ledges =>
            case checkNeighbour (IPoint.moveRight point) block_ledges of
              NONE => false
            | SOME (block,ledges) =>
              let
                (* the board topology means that ledges > 0 *)
(*GomiDebug
                val _ = ledges > 0 orelse raise Bug "nonpositive ledges"
*)
                val oldBlockLedges = findLedges blockLedges block
              in
                oldBlockLedges > ledges andalso
                Side.equal side (findSide sideStones block) andalso
                let
                  val s = fillEye board point
(*GomiDebug
                  val _ =
                      case s of
                        NONE => raise Bug "fillEye failed"
                      | SOME s => Side.equal side s orelse
                                  raise Bug "fillEye returned wrong side"
*)
                in
                  true
                end
              end
    end
(*GomiDebug
    handle Bug bug => raise Bug ("IBoard.playStoneMove:\n" ^ bug);
*)

fun playMove board s m =
    if m = IMove.PASS then playPassMove board
    else
      let
        val success = playStoneMove board s m
(*GomiDebug
        val _ = success orelse raise Bug "IBoard.playMove: move failed"
*)
      in
        ()
      end;

fun playMoves _ _ [] = ()
  | playMoves board s (m :: ms) =
    let
      val () = playMove board s m
      val s = Side.opponent s
    in
      playMoves board s ms
    end;

(* ------------------------------------------------------------------------- *)
(* Last move played.                                                         *)
(* ------------------------------------------------------------------------- *)

fun lastMoveKnown (Board {lastMove,...}) = knownLastMove lastMove;

(*GomiDebug
fun checkLastMoveKnown board =
    if lastMoveKnown board then ()
    else raise Bug "IBoard.checkLastMoveKnown";
*)

fun lastMove (Board {lastMove = lM, ...}) =
    let
      val LastMove {status = ref status, point = ref point, ...} = lM
    in
      case status of
        UnknownLastMove => NONE
      | PassLastMove => SOME IMove.PASS
      | StoneLastMove => SOME (IMove.mkStoneMove point)
    end;

fun lastMoveCaptured board =
    let
(*GomiDebug
      val () = checkLastMoveKnown board
*)
      val Board {lastMove,...} = board
      val LastMove {captured,...} = lastMove
    in
      captured
    end;

local
  fun uncaptureBlock blockLedges block point =
      let
        val block' = peekBlock blockLedges point
(*GomiDebug
        val _ = block' <> DISAPPEARING_BLOCK orelse
                raise Bug "disappearing block"
*)
      in
        if block' = EDGE_BLOCK orelse block' = block then 0
        else if block' = EMPTY_BLOCK then
          let
            val () = Array.update (blockLedges,point,block)
          in
            uncaptureStone blockLedges block point
          end
        else
          let
            val () = decrementLedges blockLedges block' 1
          in
            0
          end
      end

  and uncaptureStone blockLedges block point =
      let
        val u = uncaptureBlock blockLedges block (IPoint.moveUp point)
        val d = uncaptureBlock blockLedges block (IPoint.moveDown point)
        val l = uncaptureBlock blockLedges block (IPoint.moveLeft point)
        val r = uncaptureBlock blockLedges block (IPoint.moveRight point)
      in
        u + d + l + r + 1
      end;

  fun uncapture blockLedges sideStones empty side point =
      isEmptyPoint blockLedges point andalso
      (IIntSet.member point empty orelse
       let
         val () = storeLedges blockLedges point 0
         val stones = uncaptureStone blockLedges point point
         val () = storeSideStones sideStones point (Side.opponent side) stones
       in
         false
       end);

  fun unmergeMarkStone blockLedges undoPoint undoBlock point =
      let
        val () = Array.update (blockLedges,point,MARKED_POINT)
        val u = unmergeMarkBlock blockLedges undoPoint undoBlock
                  (IPoint.moveUp point)
        val d = unmergeMarkBlock blockLedges undoPoint undoBlock
                  (IPoint.moveDown point)
        val l = unmergeMarkBlock blockLedges undoPoint undoBlock
                  (IPoint.moveLeft point)
        val r = unmergeMarkBlock blockLedges undoPoint undoBlock
                  (IPoint.moveRight point)
      in
        u + d + l + r
      end

  and unmergeMarkBlock blockLedges undoPoint undoBlock point =
      if point = undoPoint then 0
      else
        let
          val block = peekBlock blockLedges point
(*GomiDebug
          val _ = block <> DISAPPEARING_BLOCK orelse
                  raise Bug "disappearing block"
*)
        in
          if block = EMPTY_BLOCK then 1
          else if block = undoBlock orelse block = BEING_MARKED_BLOCK then
            unmergeMarkStone blockLedges undoPoint undoBlock point
          else 0
        end;

  fun unmergeUnmarkStone blockLedges block point =
      let
        val u = unmergeUnmarkBlock blockLedges block (IPoint.moveUp point)
        val d = unmergeUnmarkBlock blockLedges block (IPoint.moveDown point)
        val l = unmergeUnmarkBlock blockLedges block (IPoint.moveLeft point)
        val r = unmergeUnmarkBlock blockLedges block (IPoint.moveRight point)
      in
        u + d + l + r + 1
      end

  and unmergeUnmarkBlock blockLedges block point =
      let
        val block' = peekBlock blockLedges point
(*GomiDebug
        val _ = block' <> DISAPPEARING_BLOCK orelse
                raise Bug "disappearing block"
        val _ = block' <> BEING_MARKED_BLOCK orelse
                raise Bug "being marked block"
*)
      in
        if block' = MARKED_BLOCK then
          let
            val () = Array.update (blockLedges,point,block)
          in
            unmergeUnmarkStone blockLedges block point
          end
        else 0
      end;

  fun unmerge blockLedges sideStones side undoPoint undoBlock block =
      let
        val ledges = unmergeMarkStone blockLedges undoPoint undoBlock block
        val () = storeLedges blockLedges block ledges
        val () = decrementLedges blockLedges undoBlock ledges
        val stones = unmergeUnmarkStone blockLedges block block
        val () = storeSideStones sideStones block side stones
        val () = decrementStones sideStones undoBlock stones
      in
        ()
      end;

  fun incLedges blockLedges point =
      let
        val block = peekBlock blockLedges point
(*GomiDebug
        val _ = block <> DISAPPEARING_BLOCK orelse raise Bug "disappearing block"
        val _ = block <> MARKED_BLOCK orelse raise Bug "marked block"
        val _ = block <> BEING_MARKED_BLOCK orelse raise Bug "being marked block"
*)
      in
        if block = EDGE_BLOCK orelse block = EMPTY_BLOCK then ()
        else incrementLedges blockLedges block 1
      end;
in
  fun undoLastMove board =
      let
        val Board
              {blockLedges,
               sideStones,
               empty,
               ko,
               lastMove = lM} = board

        val LastMove
              {status as ref stat,
               point = ref point,
               blocks,
               captured,
               ko = ref k} = lM

(*GomiDebug
        val checkUndo = Portable.randomInt UNDO_LAST_MOVE_CHECK = 0
        val checkBoard = if checkUndo then toBoard board
                         else Position.board Position.initialDefault
        val checkKo = !ko
*)

        val () =
            case stat of
              UnknownLastMove =>
              let
(*GomiDebug
                val () = raise Bug "unknown last move"
*)
              in
                ()
              end
            | PassLastMove => ()
            | StoneLastMove =>
              let
                (* find the block of the point *)
                val block = peekBlock blockLedges point
(*GomiDebug
                val _ = block <> EDGE_BLOCK orelse raise Bug "edge block"
                val _ = block <> EMPTY_BLOCK orelse raise Bug "empty block"
                val _ = block <> MARKED_BLOCK orelse raise Bug "marked block"
                val _ = block <> DISAPPEARING_BLOCK orelse
                        raise Bug "disappearing block"
                val _ = block <> BEING_MARKED_BLOCK orelse
                        raise Bug "being marked block"
*)

                val singletonBlock = block = point

                (* find the side of the last move *)
                val side = findSide sideStones block

                (* find the neighbours of the last move *)
                val up = IPoint.moveUp point
                and down = IPoint.moveDown point
                and left = IPoint.moveLeft point
                and right = IPoint.moveRight point

                (* reinstate captured stones *)
                val () = IStack.appTopDown (IIntSet.delete empty) captured
                val pointLedges = 0
                val pointLedges =
                    if uncapture blockLedges sideStones empty side up
                    then pointLedges + 1
                    else pointLedges
                val pointLedges =
                    if uncapture blockLedges sideStones empty side down
                    then pointLedges + 1
                    else pointLedges
                val pointLedges =
                    if uncapture blockLedges sideStones empty side left
                    then pointLedges + 1
                    else pointLedges
                val pointLedges =
                    if uncapture blockLedges sideStones empty side right
                    then pointLedges + 1
                    else pointLedges
                val () = if singletonBlock then ()
                         else decrementLedges blockLedges block pointLedges

                (* make the point empty *)
                val () = Array.update (blockLedges,point,EMPTY_POINT)
                val () = if singletonBlock then ()
                         else decrementStones sideStones block 1
                val () = IIntSet.add empty point

(*GomiDebug
                val _ =
                    not (mem block (IStack.toList blocks)) orelse
                    raise Bug "undo point block should not be in merged list"
                val _ =
                    not singletonBlock orelse
                    IStack.isEmpty blocks orelse
                    raise Bug "singleton block should not have merged"
*)

                (* unmerge the blocks *)
                val () =
                    IStack.appBottomUp
                      (unmerge blockLedges sideStones side point block) blocks

                (* up the ledge count of neighbouring blocks *)
                val () = incLedges blockLedges up
                val () = incLedges blockLedges down
                val () = incLedges blockLedges left
                val () = incLedges blockLedges right
              in
                ()
              end

        (* set the last move status to "unknown" *)
        val () = status := UnknownLastMove

        (* remember the ko point *)
        val () = ko := k

(*GomiDebug
        val () =
            if not checkUndo then ()
            else
              check board
              handle Bug bug =>
                raise Bug
                let
                  val ppPointStack =
                      Print.ppMap IStack.toList (Print.ppList ppBlock)
                  val b = Board.toString checkBoard
                  and uk = IPoint.toString checkKo
                  and s = toStringStatus stat
                  and p = IPoint.toString point
                  and bs = Print.toString ppPointStack blocks
                  and c = Print.toString ppPointStack captured
                  and lk = IPoint.toString k
                in
                  "board before undo =\n" ^ b ^
                  "ko before undo = " ^ uk ^ "\n" ^
                  "last move status = " ^ s ^ "\n" ^
                  "last move point = " ^ p ^ "\n" ^
                  "last move blocks = " ^ bs ^ "\n" ^
                  "last move captured = " ^ c ^ "\n" ^
                  "last move ko = " ^ lk ^ "\n" ^
                  bug
                end
*)
      in
        ()
      end
(*GomiDebug
      handle Bug bug => raise Bug ("IBoard.undoLastMove: " ^ bug)
           | e => raise Bug ("IBoard.undoLastMove: " ^ exnMessage e);
*)
end;

(* ------------------------------------------------------------------------- *)
(* Pattern matching.                                                         *)
(* ------------------------------------------------------------------------- *)

local
  fun peekBlock board p =
      let
        val pt = IPoint.fromPoint p
      in
        case peek board pt of
          SOME _ => SOME (block board pt)
        | NONE => NONE
      end;
in
  fun matchesInteger board =
      let
        fun matches patInt =
            case patInt of
              Pattern.Integer i => i
            | Pattern.Negate f => ~(matches f)
            | Pattern.Add (f1,f2) => matches f1 + matches f2
            | Pattern.Multiply (f1,f2) => matches f1 * matches f2
            | Pattern.StonesBlock p =>
              (case peekBlock board p of
                 SOME b => stones board b
               | NONE => 0)
            | Pattern.LedgesBlock p =>
              (case peekBlock board p of
                 SOME b => ledges board b
               | NONE => 0)
      in
        matches
      end;

  fun matchesSide board toMove =
      let
        fun matches patSide =
            case patSide of
              Pattern.Side s => s
            | Pattern.Opponent s => Option.map Side.opponent (matches s)
            | Pattern.SideToMove =>
              let
                val {toMove = s} = toMove
              in
                SOME s
              end
            | Pattern.SidePoint p => peek board (IPoint.fromPoint p)
      in
        matches
      end;

  fun matchesPattern board toMove =
      let
        fun matches pat =
            case pat of
              Pattern.Boolean b => b
            | Pattern.Not p => not (matches p)
            | Pattern.And (p1,p2) => matches p1 andalso matches p2
            | Pattern.Or (p1,p2) => matches p1 orelse matches p2
            | Pattern.Implies (p1,p2) => not (matches p1) orelse matches p2
            | Pattern.Iff (p1,p2) => matches p1 = matches p2
            | Pattern.LessThan (i1,i2) =>
              matchesInteger board i1 < matchesInteger board i2
            | Pattern.LessEqual (i1,i2) =>
              matchesInteger board i1 <= matchesInteger board i2
            | Pattern.IntegerEqual (i1,i2) =>
              matchesInteger board i1 = matchesInteger board i2
            | Pattern.GreaterEqual (i1,i2) =>
              matchesInteger board i1 >= matchesInteger board i2
            | Pattern.GreaterThan (i1,i2) =>
              matchesInteger board i1 > matchesInteger board i2
            | Pattern.SideEqual (s1,s2) =>
              matchesSide board toMove s1 = matchesSide board toMove s2
            | Pattern.ConnectedBlock (p1,p2) =>
              (case peekBlock board p1 of
                 NONE => false
               | SOME b1 =>
                 case peekBlock board p2 of
                   NONE => false
                 | SOME b2 => b1 = b2)
            | Pattern.Edge (e,i) =>
              case e of
                Pattern.LeftEdge => i = 0
              | Pattern.RightEdge => i = IPoint.FILES - 1
              | Pattern.TopEdge => i = IPoint.RANKS - 1
              | Pattern.BottomEdge => i = 0
      in
        matches
      end;

  val matches = matchesPattern;
end;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

val pp = Print.ppMap toBoard Board.pp;

val toString = Print.toString pp;

end
