(* ========================================================================= *)
(* IMPERATIVE TERMINAL GO POSITIONS                                          *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure IConfiguration :> IConfiguration =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* A type of terminal go positions.                                          *)
(* ------------------------------------------------------------------------- *)

datatype configuration =
    Configuration of
      {board : IBoard.board,
       status : Status.status Array.array,
       pointsScore : int ref};

fun new board =
    Configuration
      {board = board,
       status = Array.array (IPoint.POINTS,Status.Seki),
       pointsScore = ref 0};

fun load configuration =
    let
      val Configuration {board,status,pointsScore} = configuration

      fun updatePoint (point,acc) =
          let
            val (acc,stat) =
                case IBoard.peek board point of
                  SOME Side.Black => (acc + 1, Status.Stone Side.Black)
                | SOME Side.White => (acc - 1, Status.Stone Side.White)
                | NONE =>
                  case IBoard.fillEye board point of
                    SOME Side.Black => (acc + 1, Status.Eye Side.Black)
                  | SOME Side.White => (acc - 1, Status.Eye Side.White)
                  | NONE => (0,Status.Seki)

            val () = Array.update (status,point,stat)
          in
            acc
          end

      val points = List.foldl updatePoint 0 IPoint.boardPoints

      val () = pointsScore := points
    in
      ()
    end
(*GomiDebug
    handle Bug bug => raise Bug ("IConfiguration.load:\n" ^ bug);
*)

(* ------------------------------------------------------------------------- *)
(* Primitive O(1) queries.                                                   *)
(* ------------------------------------------------------------------------- *)

fun status configuration point =
    let
      val Configuration {status = s, ...} = configuration
    in
      Array.sub (s,point)
    end;

fun pointsScore configuration =
    let
      val Configuration {pointsScore = ref p, ...} = configuration
    in
      p
    end;

fun pointsGroup configuration point =
    let
      val Configuration {board,...} = configuration
    in
      case IBoard.peek board point of
        NONE => 0
      | SOME _ => IBoard.stones board (IBoard.block board point)
    end;

fun connectedGroup configuration point1 point2 =
    let
      val Configuration {board,...} = configuration
    in
      case (IBoard.peek board point1, IBoard.peek board point2) of
        (NONE,_) => false
      | (_,NONE) => false
      | (SOME Side.Black, SOME Side.White) => false
      | (SOME Side.White, SOME Side.Black) => false
      | (SOME _, SOME _) =>
        let
          val block1 = IBoard.block board point1
          and block2 = IBoard.block board point2
        in
          IBoard.equalBlock block1 block2
        end
    end;

fun sekiGroup configuration point =
    let
      val Configuration {board,...} = configuration
    in
      case IBoard.peek board point of
        NONE => false
      | SOME _ => IBoard.ledges board (IBoard.block board point) > 0
    end;

(* ------------------------------------------------------------------------- *)
(* Evaluating formulas.                                                      *)
(* ------------------------------------------------------------------------- *)

fun interpretInteger configuration =
    let
      fun interpret fmInt =
          case fmInt of
            Formula.Integer i => i
          | Formula.Negate f => ~(interpret f)
          | Formula.Add (f1,f2) => interpret f1 + interpret f2
          | Formula.Multiply (f1,f2) => interpret f1 * interpret f2
          | Formula.PointsScore => pointsScore configuration
          | Formula.PointsGroup p =>
            pointsGroup configuration (IPoint.fromPoint p)
    in
      interpret
    end;

fun interpretStatus configuration =
    let
      fun interpret fmStat =
          case fmStat of
            Formula.Status s => s
          | Formula.StatusPoint p => status configuration (IPoint.fromPoint p)
    in
      interpret
    end;

fun interpretSide configuration =
    let
      fun interpret fmSide =
          case fmSide of
            Formula.Side s => s
          | Formula.Opponent s => Option.map Side.opponent (interpret s)
          | Formula.SideStatus s =>
            Status.destTerritory (interpretStatus configuration s)
          | Formula.SideStoneStatus s =>
            Status.destStone (interpretStatus configuration s)
          | Formula.SideEyeStatus s =>
            Status.destEye (interpretStatus configuration s)
    in
      interpret
    end;

fun interpretFormula configuration =
    let
      fun interpret fm =
          case fm of
            Formula.Boolean b => b
          | Formula.Not f => not (interpret f)
          | Formula.And (f1,f2) => interpret f1 andalso interpret f2
          | Formula.Or (f1,f2) => interpret f1 orelse interpret f2
          | Formula.Implies (f1,f2) => not (interpret f1) orelse interpret f2
          | Formula.Iff (f1,f2) => interpret f1 = interpret f2
          | Formula.LessThan (i1,i2) =>
            interpretInteger configuration i1 <
            interpretInteger configuration i2
          | Formula.LessEqual (i1,i2) =>
            interpretInteger configuration i1 <=
            interpretInteger configuration i2
          | Formula.IntegerEqual (i1,i2) =>
            interpretInteger configuration i1 =
            interpretInteger configuration i2
          | Formula.GreaterEqual (i1,i2) =>
            interpretInteger configuration i1 >=
            interpretInteger configuration i2
          | Formula.GreaterThan (i1,i2) =>
            interpretInteger configuration i1 >
            interpretInteger configuration i2
          | Formula.StatusEqual (s1,s2) =>
            interpretStatus configuration s1 =
            interpretStatus configuration s2
          | Formula.StatusMember (s,ss) =>
            StatusSet.member (interpretStatus configuration s) ss
          | Formula.SideEqual (s1,s2) =>
            interpretSide configuration s1 =
            interpretSide configuration s2
          | Formula.ConnectedGroup (p1,p2) =>
            let
              val pt1 = IPoint.fromPoint p1
              and pt2 = IPoint.fromPoint p2
            in
              connectedGroup configuration pt1 pt2
            end
          | Formula.SekiGroup p =>
            sekiGroup configuration (IPoint.fromPoint p)
    in
      interpret
    end;

val interpret = interpretFormula;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun toBoard configuration =
    let
      fun peekFn pt =
          Status.destStone (status configuration (IPoint.fromPoint pt))
    in
      Board.tabulate IPoint.DIMENSIONS peekFn
    end;

val pp = Print.ppMap toBoard Board.pp;

val toString = Print.toString pp;

end
