(* ========================================================================= *)
(* IMPERATIVE FORMULA PROBABILITY INFERENCE                                  *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure IInference :> IInference =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Imperative formula probability inference.                                 *)
(* ------------------------------------------------------------------------- *)

datatype inference =
    Inference of
      {estimate : Database.estimate ref};

fun new () =
    let
      val estimate = ref Database.emptyEstimate
    in
      Inference {estimate = estimate}
    end;

fun size inference =
    let
      val Inference {estimate = ref est, ...} = inference
    in
      FormulaMap.size est
    end;

fun estimate inference =
    let
      val Inference {estimate = ref est, ...} = inference
    in
      est
    end;

fun setEstimate inference est =
    let
      val Inference {estimate,...} = inference

      val () = estimate := est
    in
      ()
    end;

(* ------------------------------------------------------------------------- *)
(* Inferring formula probabilities (from the estimates).                     *)
(* ------------------------------------------------------------------------- *)

fun probabilityOption prob =
    case prob of
      SOME p => p
    | NONE => Probability.uniformDistribution;

fun inferred inference fm =
    let
      val Inference {estimate as ref est, ...} = inference
    in
      probabilityOption (FormulaMap.peek est fm)
    end;

(* ------------------------------------------------------------------------- *)
(* The effect of updating the estimates on the inferred probabilities.       *)
(* ------------------------------------------------------------------------- *)

local
  fun addUpdate (_,p) = p;

  fun removeUpdate (_,p) =
      case p of
        SOME _ => NONE
      | NONE => SOME Probability.uniformDistribution;
in
  fun updateEstimate inference est' =
      let
        val Inference {estimate,...} = inference
        val ref est = estimate
        val add = FormulaMap.mapPartial addUpdate est'
        and remove = FormulaMap.mapPartial removeUpdate est'
        val est = FormulaMap.differenceDomain est remove
        val est = FormulaMap.unionDomain est add
        val () = estimate := est
      in
        ()
      end;
end;

fun inferredAfter inference est' fm =
    case FormulaMap.peek est' fm of
      SOME p => probabilityOption p
    | NONE => inferred inference fm;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun toString inference = "Inference{" ^ Int.toString (size inference) ^ "}";

val pp = Print.ppMap toString Print.ppString;

end
