(* ========================================================================= *)
(* IMPERATIVE INTEGER SETS WITH CONSTANT TIME ADD AND RANDOM DELETE          *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure IIntSet :> IIntSet =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Constants.                                                                *)
(* ------------------------------------------------------------------------- *)

val ABSENT = ~1;

(* ------------------------------------------------------------------------- *)
(* A type of imperative integer sets.                                        *)
(* ------------------------------------------------------------------------- *)

datatype set =
    Set of
      {items : int IStack.stack,
       indexes : int Array.array};

fun empty n =
    Set
      {items = IStack.empty {maxSize = n, defaultItem = 0},
       indexes = Array.array (n,ABSENT)};

fun size (Set {items,...}) = IStack.size items;

fun member x (Set {indexes,...}) = Array.sub (indexes,x) <> ABSENT;

fun clone (Set {items,indexes}) =
    Set {items = IStack.clone items, indexes = cloneArray indexes};

fun copy src dst =
    let
      val Set {items = srcItems, indexes = srcIndexes} = src
      and Set {items = dstItems, indexes = dstIndexes} = dst
      val () = IStack.copy srcItems dstItems
      val () = Array.copy {src = srcIndexes, dst = dstIndexes, di = 0}
    in
      ()
    end;

fun fold f b (Set {items,...}) = IStack.foldTopDown f b items;

fun app f set = fold (fn (x,()) => f x) () set;

fun add set x =
    let
(*GomiDebug
      val _ = not (member x set) orelse raise Bug "IIntSet.add: already member"
*)
      val Set {items,indexes} = set
      val s = IStack.size items
      val () = IStack.push items x
      val () = Array.update (indexes,x,s)
    in
      ()
    end;

fun delete set x =
    let
(*GomiDebug
      val _ = member x set orelse raise Bug "IIntSet.delete: not a member"
*)
      val Set {items,indexes} = set
      val s = IStack.size items - 1
      val i = Array.sub (indexes,x)
      val () = Array.update (indexes,x,ABSENT)
    in
      if i = s then
        let
          val () = IStack.pop items
        in
          ()
        end
      else
        let
          val y = IStack.sub items s
          val () = IStack.remove items i
          val () = Array.update (indexes,y,i)
        in
          ()
        end
    end;

fun toList (Set {items,...}) = IStack.toList items;

(* ------------------------------------------------------------------------- *)
(* Accessing elements by their index in the set,                             *)
(* where indexes are in the range [0,size).                                  *)
(* ------------------------------------------------------------------------- *)

fun sub (Set {items,...}) i = IStack.sub items i;

fun swap (Set {items,indexes}) i j =
    if i = j then ()
    else
      let
        val () = IStack.swap items i j
        val () = Array.update (indexes, IStack.sub items i, i)
        val () = Array.update (indexes, IStack.sub items j, j)
      in
        ()
      end;

(* ------------------------------------------------------------------------- *)
(* Generating random values.                                                 *)
(* ------------------------------------------------------------------------- *)

fun random set =
    let
      val n = size set
(*GomiDebug
      val _ = n > 0 orelse raise Bug "IIntSet.random: empty"
*)
    in
      sub set (Portable.randomInt n)
    end;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun pp ppI = Print.ppMap (fn Set {items,...} => items) (IStack.pp ppI);

val toString = Print.toString (pp Print.ppInt);

end
