(* ========================================================================= *)
(* IMPERATIVE GO KNOWLEDGE                                                   *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure IKnowledge :> IKnowledge =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Constants.                                                                *)
(* ------------------------------------------------------------------------- *)

val MAXIMUM_MOVES = 3 * Dimensions.numPoints IPoint.DIMENSIONS;

(* ------------------------------------------------------------------------- *)
(* A type of pattern database.                                               *)
(* ------------------------------------------------------------------------- *)

datatype database =
    Database of
      {formulas : IInference.estimate};

val emptyDB =
    let
      val formulas = IInference.emptyEstimate
    in
      Database
        {formulas = formulas}
    end;

fun estimateDB (Database _) (_ : IPattern.set) : IInference.estimate =
    raise Bug "IKnowledge.estimateDB";

fun estimateAfterDB (Database _)
                    (_ : IPattern.set)
                    (_ : IInference.estimate)
                    (_ : IPattern.update) : IInference.update =
    raise Bug "IKnowledge.estimateAfterDB";

fun updateDB (Database _)
             (_ : IPattern.set)
             (_ : IConfiguration.configuration) : database =
    raise Bug "IKnowledge.updateDB";

(* ------------------------------------------------------------------------- *)
(* An imperative type of go knowledge.                                       *)
(* ------------------------------------------------------------------------- *)

datatype knowledge =
    Knowledge of
      {database : database ref,
       winFormulas : Formula.formula Side.sides,
       position : Position.position ref,
       board : IBoard.board,
       moves : IMove.move IStack.stack,
       patterns : IPattern.board,
       inference : IInference.inference,
       configuration : IConfiguration.configuration};

fun new posn =
    let
      val database = ref emptyDB

      val komi = Position.komi posn

      val winFormulas =
          {black = Formula.isBlackWin komi,
           white = Formula.isWhiteWin komi}

      val position = ref posn

      val board = IBoard.fromBoard (Position.board posn)

      val moves = IStack.empty {maxSize = MAXIMUM_MOVES,
                                defaultItem = IMove.NO_MOVE}

      val sampleBoard = IBoard.clone board
      and sampleToMove = ref (Position.toMove posn)

      val patterns = IPattern.new {board = sampleBoard, toMove = sampleToMove}

      val inference = IInference.new ()

      val configuration = IConfiguration.new sampleBoard
    in
      Knowledge
        {database = database,
         winFormulas = winFormulas,
         position = position,
         board = board,
         moves = moves,
         patterns = patterns,
         inference = inference,
         configuration = configuration}
    end;

fun position (Knowledge {position = ref posn, ...}) = posn;

(* ------------------------------------------------------------------------- *)
(* Gaining knowledge through sample games.                                   *)
(* ------------------------------------------------------------------------- *)

fun sampleGameMoves database winFormulas moves patterns inference =
    let
      val {board,toMove} = IPattern.dest patterns
      val pats = IPattern.patterns patterns

      fun testMove winFm est (move,set) =
          case IPattern.patternsAfter patterns move of
            NONE => set
          | SOME patUpdate =>
            let
              val estUpdate = estimateAfterDB database pats est patUpdate
              val winProb = IInference.inferredAfter inference estUpdate winFm
            in
              Probability.addSet set (move,winProb)
            end

      fun playMoves quota passes =
          if quota <= 0 orelse passes >= 2 then ()
          else
            let
              val winFm = Side.pickSides winFormulas (!toMove)
              val est = IInference.estimate inference
              val empty = IBoard.empty board
              val set = Probability.emptySet
              val set = IIntSet.fold (testMove winFm est) set empty
              val move = Probability.maxSampleSet set
              val () = IStack.push moves move
              val patUpdate = IPattern.playMove patterns move
              val estUpdate = estimateAfterDB database pats est patUpdate
              val () = IInference.updateEstimate inference estUpdate
              val quota = quota - 1
              val passes = if move = IMove.PASS then passes + 1 else 0
            in
              playMoves quota passes
            end

      val quota = MAXIMUM_MOVES - IStack.size moves

      val passes = 0

      val () = playMoves quota passes
    in
      ()
    end;

(*GomiDebug
val blackWinCount = ref 0;
val whiteWinCount = ref 0;
val drawCount = ref 0;
*)

fun sampleGame knowledge =
    let
      val Knowledge
            {database,
             winFormulas,
             position = ref posn,
             board,
             moves,
             patterns,
             inference,
             configuration} = knowledge

      val () = IStack.reset moves

      val {board = patternsBoard, toMove = patternsToMove} =
          IPattern.dest patterns

      val () = IBoard.copy board patternsBoard
      and () = patternsToMove := Position.toMove posn

      val () = IPattern.load patterns
      val pats = IPattern.patterns patterns

      val ref db = database

      val est = estimateDB db pats

      val () = IInference.setEstimate inference est

      val () = sampleGameMoves db winFormulas moves patterns inference

      val () = IConfiguration.load configuration

      val allPatterns = IPattern.allPatterns patterns

      val () = database := updateDB db allPatterns configuration

(*GomiDebug
      val points = IConfiguration.pointsScore configuration

      val score = Score.fromPoints (Position.komi posn) points

      val () =
          case score of
            Score.Win (Side.Black,_) => blackWinCount := !blackWinCount + 1
          | Score.Win (Side.White,_) => whiteWinCount := !whiteWinCount + 1
          | Score.Draw => drawCount := !drawCount + 1

      val () =
          if (!blackWinCount + !whiteWinCount + !drawCount) mod 1000 <> 0 then ()
          else
            trace ("IKnowledge.sampleGame: winCount = " ^
                   "black: " ^ Int.toString (!blackWinCount) ^
                   ", white: " ^ Int.toString (!whiteWinCount) ^
                   ", draw: " ^ Int.toString (!drawCount) ^ "\n")
*)
    in
      ()
    end
(*GomiDebug
    handle Bug bug => raise Bug ("IKnowledge.sampleGame: " ^ bug)
         | e => raise Bug ("IKnowledge.sampleGame: " ^ exnMessage e);
*)

fun sampleMoves (Knowledge {moves,...}) =
    map IMove.toMove (rev (IStack.toList moves));

fun sampleBoard (Knowledge {configuration,...}) =
    IConfiguration.toBoard configuration;

(* ------------------------------------------------------------------------- *)
(* Moves.                                                                    *)
(* ------------------------------------------------------------------------- *)

fun playMove knowledge move =
    let
      val Knowledge {position,board,...} = knowledge
      val ref posn = position
      val toMove = Position.toMove posn
      val () = position := Position.playMove posn move
      val () = IBoard.playMove board toMove (IMove.fromMove move)
    in
      ()
    end;

fun generateMove _ = raise Bug "IKnowledge.generateMove";

(* ------------------------------------------------------------------------- *)
(* Score.                                                                    *)
(* ------------------------------------------------------------------------- *)

fun score knowledge =
    let
      val () = sampleGame knowledge

      val Knowledge {configuration,...} = knowledge

      val points = IConfiguration.pointsScore configuration

      val komi = Position.komi (position knowledge)
    in
      Score.fromPoints komi points
    end;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

val pp = Parser.ppMap position Position.pp;

val toString = Parser.toString pp;

end
