(* ========================================================================= *)
(* IMPERATIVE GO BOARD PATTERNS                                              *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure IPattern :> IPattern =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Pattern operations.                                                       *)
(* ------------------------------------------------------------------------- *)

fun activeSubset iboard tM =
    let
      val isActive = IBoard.matches iboard {toMove = tM}
    in
      PatternSet.filter isActive
    end;

fun weakestSubset pats =
    let
      fun chuck pat =
          PatternSet.exists (Pattern.strictlyStrongerNormalized pat) pats
    in
      PatternSet.filter (not o chuck) pats
    end;

(* ------------------------------------------------------------------------- *)
(* Imperative go board patterns.                                             *)
(* ------------------------------------------------------------------------- *)

type boardInfo =
     {board : IBoard.board,
      toMove : Side.side ref,
      allPatterns : Database.patterns};

datatype board =
    Board of
      {allPatterns : Database.patterns ref,
       seenPatterns : Database.patterns ref,
       iboard : IBoard.board,
       toMove : Side.side ref,
       activePatterns : Database.patterns ref,
       weakestPatterns : Database.patterns ref};

fun new info =
    let
      val {board = iboard, toMove as ref tM, allPatterns = all} = info
      val active = activeSubset iboard tM all
      val weakest = weakestSubset active
    in
      Board
        {allPatterns = ref all,
         seenPatterns = ref active,
         iboard = iboard,
         toMove = toMove,
         activePatterns = ref active,
         weakestPatterns = ref weakest}
    end;

fun copy board1 board2 =
    let
      val Board
            {allPatterns = ref all1,
             seenPatterns = ref seen1,
             iboard = iboard1,
             toMove = ref tM1,
             activePatterns = ref active1,
             weakestPatterns = ref weakest1} = board1

      and Board
            {allPatterns = allPatterns2,
             seenPatterns = seenPatterns2,
             iboard = iboard2,
             toMove = toMove2,
             activePatterns = activePatterns2,
             weakestPatterns = weakestPatterns2} = board2

      val () = allPatterns2 := all1
      and () = seenPatterns2 := seen1
      and () = IBoard.copy iboard1 iboard2
      and () = toMove2 := tM1
      and () = activePatterns2 := active1
      and () = weakestPatterns2 := weakest1
    in
      ()
    end;

fun clone board =
    let
      val Board
            {allPatterns = ref all,
             seenPatterns = ref seen,
             iboard,
             toMove = ref tM,
             activePatterns = ref active,
             weakestPatterns = ref weakest} = board
    in
      Board
        {allPatterns = ref all,
         seenPatterns = ref seen,
         iboard = IBoard.clone iboard,
         toMove = ref tM,
         activePatterns = ref active,
         weakestPatterns = ref weakest}
    end;

fun dest board =
    let
      val Board {iboard, toMove, allPatterns = ref all, ...} = board
    in
      {board = iboard, toMove = toMove, allPatterns = all}
    end;

fun updateAllPatterns board update =
    let
      val Board
            {allPatterns,
             seenPatterns,
             iboard,
             toMove = ref tM,
             activePatterns,
             weakestPatterns} = board

      val ref all = allPatterns
      and ref seen = seenPatterns
      and ref active = activePatterns
      val {add,remove} = update

      val activeAdd = activeSubset iboard tM add
      val activeUpdate = {add = activeAdd, remove = remove}

      val all = Database.updatePatterns all update
      and seen = Database.updatePatterns seen activeUpdate
      and active = Database.updatePatterns active activeUpdate
      val weakest = weakestSubset active

      val () = allPatterns := all
      and () = seenPatterns := seen
      and () = activePatterns := active
      and () = weakestPatterns := weakest
    in
      ()
    end;

fun seenPatterns (Board {seenPatterns = ref seen, ...}) = seen;

fun patterns (Board {weakestPatterns = ref weakest, ...}) = weakest;

(* ------------------------------------------------------------------------- *)
(* The effect of playing a move on the board patterns.                       *)
(* ------------------------------------------------------------------------- *)

fun patternsAfter board move =
    let
      val Board
            {allPatterns = ref all,
             iboard,
             toMove = ref tM,
             weakestPatterns = ref weakest,
             ...} = board

      val ok =
          if move = IMove.PASS then (IBoard.playPassMove iboard; true)
          else IBoard.playSensibleStoneMove iboard tM move
    in
      if not ok then NONE
      else
        let
          val active = activeSubset iboard (Side.opponent tM) all
          val weakest' = weakestSubset active
          val () = IBoard.undoLastMove iboard
        in
          SOME (Database.subtractPatterns weakest' weakest)
        end
    end;

fun playMove board move =
    let
      val Board
            {allPatterns = ref all,
             seenPatterns as ref seen,
             iboard,
             toMove as ref tM,
             activePatterns,
             weakestPatterns as ref weakest} = board

      val () = IBoard.playMove iboard tM move
      val tM = Side.opponent tM
      val () = toMove := tM
      val active = activeSubset iboard tM all
      val () = activePatterns := active
      val seen = PatternSet.union seen active
      val () = seenPatterns := seen
      val weakest' = weakestSubset active
    in
      Database.subtractPatterns weakest' weakest
    end;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

val pp = Print.ppMap (fn Board {iboard,...} => iboard) IBoard.pp;

val toString = Print.toString pp;

end
