(* ========================================================================= *)
(* POINTS ON A FIXED-SIZE BOARD.                                             *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure IPoint :> IPoint =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Board dimensions.                                                         *)
(* ------------------------------------------------------------------------- *)

val DIMENSIONS = Dimensions.default;

val {files = FILES, ranks = RANKS} = DIMENSIONS;

(*GomiDebug
val () = if FILES >= 2 then () else raise Bug "FILES must be at least 2";
val () = if RANKS >= 2 then () else raise Bug "RANKS must be at least 2";
*)

(* ------------------------------------------------------------------------- *)
(* Points as integers.                                                       *)
(* ------------------------------------------------------------------------- *)

type point = int;

val NO_POINT = ~1;

val ROW = FILES + 1;

val POINTS = ROW * RANKS + 1;

fun equal (p1 : point) p2 = p1 = p2;

fun isValid point = 0 <= point andalso point < POINTS;

fun isEdge point =
    let
(*GomiDebug
      val _ = isValid point orelse raise Bug "IPoint.isEdge: invalid"
*)
    in
      point mod ROW = 0
    end;

fun onBoard point = not (isEdge point);

local
  val TOP_EDGE_POINT = POINTS - 1;
  val SECOND_TOP_EDGE_POINT = TOP_EDGE_POINT - ROW;
in
  fun moveUp point =
    let
(*GomiDebug
      val _ = onBoard point orelse raise Bug "IPoint.moveUp: edge point"
*)
      val point =
          if point > SECOND_TOP_EDGE_POINT then TOP_EDGE_POINT
          else point + ROW
(*GomiDebug
      val _ = isValid point orelse raise Bug "IPoint.moveUp: invalid result"
*)
    in
      point
    end;
end;

local
  val BOTTOM_EDGE_POINT = 0;
  val SECOND_BOTTOM_EDGE_POINT = BOTTOM_EDGE_POINT + ROW;

in
  fun moveDown point =
    let
(*GomiDebug
      val _ = onBoard point orelse raise Bug "IPoint.moveDown: edge point"
*)
      val point =
          if point < SECOND_BOTTOM_EDGE_POINT then BOTTOM_EDGE_POINT
          else point - ROW
(*GomiDebug
      val _ = isValid point orelse raise Bug "IPoint.moveDown: invalid result"
*)
    in
      point
    end;
end;

fun moveLeft point =
    let
(*GomiDebug
      val _ = onBoard point orelse raise Bug "IPoint.moveLeft: edge point"
*)
      val point = point - 1
(*GomiDebug
      val _ = isValid point orelse raise Bug "IPoint.moveLeft: invalid result"
*)
    in
      point
    end;

fun moveRight point =
    let
(*GomiDebug
      val _ = onBoard point orelse raise Bug "IPoint.moveRight: edge point"
*)
      val point = point + 1
(*GomiDebug
      val _ = isValid point orelse raise Bug "IPoint.moveRight: invalid result"
*)
    in
      point
    end;

val allPoints = List.tabulate (POINTS,I);

fun fromPoint (Point.Point {file,rank}) =
    let
(*GomiDebug
      val _ = 0 <= file orelse raise Bug "IPoint.fromPoint: negative file"
      val _ = file < FILES orelse raise Bug "IPoint.fromPoint: large file"
      val _ = 0 <= rank orelse raise Bug "IPoint.fromPoint: negative rank"
      val _ = rank < RANKS orelse raise Bug "IPoint.fromPoint: large rank"
*)
    in
      ROW * rank + file + 1
   end;

fun toPoint point =
    let
(*GomiDebug
      val _ = onBoard point orelse raise Bug "IPoint.toPoint: edge point"
*)
      val point = point - 1
      val file = point mod ROW
      val rank = point div ROW
    in
      Point.Point {file = file, rank = rank}
    end;

val boardPoints = List.filter onBoard allPoints;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun pp point =
    if point = NO_POINT then Print.ppString "<no-point>"
    else if isEdge point then Print.ppString "<edge>"
    else Point.pp (toPoint point);

val toString = Print.toString pp

end
