(* ========================================================================= *)
(* IMPERATIVE SAMPLE GAMES                                                   *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure ISample :> ISample =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Constants.                                                                *)
(* ------------------------------------------------------------------------- *)

val MAXIMUM_MOVES = 3 * Dimensions.numPoints IPoint.DIMENSIONS;

(* ------------------------------------------------------------------------- *)
(* An imperative type of game sampler.                                       *)
(* ------------------------------------------------------------------------- *)

datatype sample =
    Sample of
      {position : Position.position,
       initialPatterns : IPattern.board,
       winFormulas : Formula.formula Side.sides,
       moves : IMove.move IStack.stack,
       patterns : IPattern.board,
       inference : IInference.inference,
       configuration : IConfiguration.configuration};

fun new position database =
    let
      val board = IBoard.fromBoard (Position.board position)

      val toMove = Position.toMove position

      val allPatterns = Database.patterns database

      val patterns =
          IPattern.new
            {board = board, toMove = ref toMove, allPatterns = allPatterns}

      val initialPatterns = IPattern.clone patterns

      val komi = Position.komi position

      val winFormulas =
          {black = Formula.isBlackWin komi,
           white = Formula.isWhiteWin komi}

      val moves = IStack.empty {maxSize = MAXIMUM_MOVES,
                                defaultItem = IMove.NO_MOVE}

      val inference = IInference.new ()

      val configuration = IConfiguration.new board
    in
      Sample
        {position = position,
         initialPatterns = initialPatterns,
         winFormulas = winFormulas,
         moves = moves,
         patterns = patterns,
         inference = inference,
         configuration = configuration}
    end
(*GomiDebug
    handle Bug bug => raise Bug ("ISample.new: " ^ bug)
         | e => raise Bug ("ISample.new: " ^ exnMessage e);
*)

fun position (Sample {position = p, ...}) = p;

(* ------------------------------------------------------------------------- *)
(* Playing sample games.                                                     *)
(* ------------------------------------------------------------------------- *)

fun sampleGame database winFormulas moves patterns inference =
    let
      val {board,toMove,...} = IPattern.dest patterns

      fun testMove winFm pats est (move,set) =
          case IPattern.patternsAfter patterns move of
            NONE => set
          | SOME patUpdate =>
            let
              val estUpdate = Database.estimateAfter database pats est patUpdate
              val winProb = IInference.inferredAfter inference estUpdate winFm
            in
              Probability.addSet set (move,winProb)
            end

      fun playMoves quota passes =
          if quota <= 0 orelse passes >= 2 then ()
          else
            let
              val winFm = Side.pickSides winFormulas (!toMove)
              val pats = IPattern.patterns patterns
              val est = IInference.estimate inference
              val empty = IBoard.empty board
              val set = Probability.emptySet
              val set = IIntSet.fold (testMove winFm pats est) set empty
              val set = testMove winFm pats est (IMove.PASS,set)
              val move = Probability.randomMaxSet set
              val () = IStack.push moves move
              val patUpdate = IPattern.playMove patterns move
              val estUpdate = Database.estimateAfter database pats est patUpdate
              val () = IInference.updateEstimate inference estUpdate
              val quota = quota - 1
              val passes = if move = IMove.PASS then passes + 1 else 0
            in
              playMoves quota passes
            end

      val quota = MAXIMUM_MOVES - IStack.size moves

      val passes = 0

      val () = playMoves quota passes
    in
      ()
    end;

(*GomiDebug
val blackWinCount = ref 0;
val whiteWinCount = ref 0;
val drawCount = ref 0;
*)

fun sample sampleData database =
    let
      val Sample
            {position,
             initialPatterns,
             winFormulas,
             moves,
             patterns,
             inference,
             configuration} = sampleData

      val () = IStack.reset moves

      val () = IPattern.copy initialPatterns patterns

      val pats = IPattern.patterns patterns

      val est = Database.estimate database pats

      val () = IInference.setEstimate inference est

      val () = sampleGame database winFormulas moves patterns inference

      val () = IConfiguration.load configuration

      val seenPats = IPattern.seenPatterns patterns

      val formulas = IConfiguration.interpret configuration

      val (database,allPatsUpdate) = Database.learn database seenPats formulas

      val () = IPattern.updateAllPatterns initialPatterns allPatsUpdate

(*GomiDebug
      val points = IConfiguration.pointsScore configuration

      val score = Score.fromPoints (Position.komi position) points

      val () =
          case score of
            Score.Win (Side.Black,_) => blackWinCount := !blackWinCount + 1
          | Score.Win (Side.White,_) => whiteWinCount := !whiteWinCount + 1
          | Score.Draw => drawCount := !drawCount + 1

      val () =
          if (!blackWinCount + !whiteWinCount + !drawCount) mod 1000 <> 0 then ()
          else
            trace ("ISample.sample: winCount = " ^
                   "black: " ^ Int.toString (!blackWinCount) ^
                   ", white: " ^ Int.toString (!whiteWinCount) ^
                   ", draw: " ^ Int.toString (!drawCount) ^ "\n")
*)
    in
      database
    end
(*GomiDebug
    handle Bug bug => raise Bug ("ISample.sample: " ^ bug)
         | e => raise Bug ("ISample.sample: " ^ exnMessage e);
*)

(* ------------------------------------------------------------------------- *)
(* Examining the state of a sample game.                                     *)
(* ------------------------------------------------------------------------- *)

fun moves (Sample {moves = m, ...}) = m;

fun patterns (Sample {patterns = p, ...}) = p;

fun configuration (Sample {configuration = c, ...}) = c;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

val pp = Print.ppMap position Position.pp;

val toString = Print.toString pp;

end
