(* ========================================================================= *)
(* IMPERATIVE STACKS                                                         *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure IStack :> IStack =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* A type of imperative stacks.                                              *)
(* ------------------------------------------------------------------------- *)

datatype 'a stack = Stack of {size : int ref, items : 'a Array.array};

fun empty {maxSize,defaultItem} =
    Stack
      {size = ref 0,
       items = Array.array (maxSize,defaultItem)};

fun size (Stack {size = ref s, ...}) = s;

fun isEmpty stack = size stack = 0;

fun top (Stack {size = ref s, items}) =
    let
(*GomiDebug
      val _ = s > 0 orelse raise Bug "IStack.top: empty"
*)
    in
      Array.sub (items, s - 1)
    end;

fun nth (Stack {size = ref s, items}) n =
    let
(*GomiDebug
      val _ = s > n orelse raise Bug "IStack.nth: no nth element"
*)
    in
      Array.sub (items, s - (n + 1))
    end;

fun push (Stack {size,items}) item =
    let
      val ref s = size
(*GomiDebug
      val _ = s < Array.length items orelse
              raise Bug "IStack.push: already full"
*)
      val () = size := s + 1
      val () = Array.update (items,s,item)
    in
      ()
    end;

fun pop (Stack {size,...}) =
    let
(*GomiDebug
      val _ = !size > 0 orelse raise Bug "IStack.pop: empty"
*)
      val () = size := !size - 1
    in
      ()
    end;

fun popN (Stack {size,...}) n =
    let
(*GomiDebug
      val _ = !size > n orelse raise Bug "IStack.popN: less than n elements"
*)
      val () = size := !size - n
    in
      ()
    end;

fun popTop (Stack {size as ref s, items}) =
    let
(*GomiDebug
      val _ = s > 0 orelse raise Bug "IStack.popTop: empty"
*)
      val s = s - 1
      val () = size := s
    in
      Array.sub (items,s)
    end;

fun reset (Stack {size,...}) = size := 0;

fun clone (Stack {size = ref s, items}) =
    Stack {size = ref s, items = cloneArray items};

fun copy src dst =
    let
      val Stack {size = srcSize, items = srcItems} = src
      and Stack {size = dstSize, items = dstItems} = dst
(*GomiDebug
      val _ = srcItems <> dstItems orelse
              raise Bug "IStack.copy: same array"
      val _ = Array.length srcItems = Array.length dstItems orelse
              raise Bug "IStack.copy: different maxSizes"
*)
      val () = dstSize := !srcSize
      val () = Array.copy {src = srcItems, dst = dstItems, di = 0}
    in
      ()
    end;

fun foldTopDown f b (Stack {size = ref s, items}) =
    let
      fun fld 0 acc = acc
        | fld i acc =
          let
            val i = i - 1
            val item_i = Array.sub (items,i)
          in
            fld i (f (item_i,acc))
          end
    in
      fld s b
    end;

fun foldBottomUp f b (Stack {size = ref s, items}) =
    let
      fun fld i acc =
          if i = s then acc
          else
            let
              val item_i = Array.sub (items,i)
            in
              fld (i + 1) (f (item_i,acc))
            end
    in
      fld 0 b
    end;

fun appTopDown f stack = foldTopDown (fn (x,()) => f x) () stack;

fun appBottomUp f stack = foldBottomUp (fn (x,()) => f x) () stack;

fun all pred (Stack {size = ref s, items}) =
    let
      fun f 0 = true
        | f i =
          let
            val i = i - 1
            val item_i = Array.sub (items,i)
          in
            pred item_i andalso f i
          end
    in
      f s
    end;

fun exists pred (Stack {size = ref s, items}) =
    let
      fun f 0 = false
        | f i =
          let
            val i = i - 1
            val item_i = Array.sub (items,i)
          in
            pred item_i orelse f i
          end
    in
      f s
    end;

fun toList stack = List.tabulate (size stack, nth stack);

(* ------------------------------------------------------------------------- *)
(* Accessing elements by their index in the stack.                           *)
(* Indexes are in the range [0,size), where 0 is the bottom of the stack.    *)
(* ------------------------------------------------------------------------- *)

fun sub stack i =
    let
(*GomiDebug
      val Stack {size = ref s, ...} = stack
      val _ = i < s orelse raise Bug "IStack.sub: out of range"
*)
      val Stack {items,...} = stack
    in
      Array.sub (items,i)
    end;

fun update stack i item =
    let
(*GomiDebug
      val Stack {size = ref s, ...} = stack
      val _ = i < s orelse raise Bug "IStack.update: i out of range"
*)
      val Stack {items,...} = stack
      val () = Array.update (items,i,item)
    in
      ()
    end;

fun duplicate stack i j =
    let
(*GomiDebug
      val Stack {size = ref s, ...} = stack
      val _ = i < s orelse raise Bug "IStack.duplicate: i out of range"
      val _ = j < s orelse raise Bug "IStack.duplicate: j out of range"
*)
    in
      if i = j then ()
      else
        let
          val Stack {items,...} = stack
          val item = Array.sub (items,i)
          val () = Array.update (items,j,item)
        in
          ()
        end
    end;

fun swap stack i j =
    let
(*GomiDebug
      val Stack {size = ref s, ...} = stack
      val _ = i < s orelse raise Bug "IStack.swap: i out of range"
      val _ = j < s orelse raise Bug "IStack.swap: j out of range"
*)
    in
      if i = j then ()
      else
        let
          val Stack {items,...} = stack
          val item_i = Array.sub (items,i)
          and item_j = Array.sub (items,j)
          val () = Array.update (items,i,item_j)
          val () = Array.update (items,j,item_i)
        in
          ()
        end
    end;

fun remove stack i =
    let
      val Stack {size as ref s, ...} = stack
(*GomiDebug
      val _ = s > 0 orelse raise Bug "IStack.remove: empty"
*)
      val s = s - 1
      val () = duplicate stack s i
      val () = size := s
    in
      ()
    end;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun pp ppA = Print.ppMap toList (Print.ppList ppA);

end
