(* ========================================================================= *)
(* PATTERNS OVER GO POSITIONS                                                *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature Pattern =
sig

(* ------------------------------------------------------------------------- *)
(* Board edges.                                                              *)
(* ------------------------------------------------------------------------- *)

datatype edge = LeftEdge | RightEdge | TopEdge | BottomEdge

val compareEdge : edge * edge -> order

(* ------------------------------------------------------------------------- *)
(* Integers associated with go positions.                                    *)
(* ------------------------------------------------------------------------- *)

datatype integer =
  (* Primitive integer operations *)
    Integer of int
  | Negate of integer
  | Add of integer * integer
  | Multiply of integer * integer

  (* The number of stones in a block *)
  | StonesBlock of Point.point

  (* The number of liberty edges (ledges) from a block *)
  | LedgesBlock of Point.point

val compareInteger : integer * integer -> order

val isValidInteger : Dimensions.dimensions -> integer -> bool

val transformInteger : Symmetry.symmetry -> integer -> integer

(* ------------------------------------------------------------------------- *)
(* Sides in go positions.                                                    *)
(* ------------------------------------------------------------------------- *)

datatype side =
  (* Primitive status operations *)
    Side of Side.side option
  | Opponent of side

  (* Side to move *)
  | SideToMove

  (* Side of a point *)
  | SidePoint of Point.point

val compareSide : side * side -> order

val isValidSide : Dimensions.dimensions -> side -> bool

val transformSide : Symmetry.symmetry -> side -> side

val blackSide : side

val whiteSide : side

val noSide : side

(* ------------------------------------------------------------------------- *)
(* Patterns in go positions.                                                 *)
(* ------------------------------------------------------------------------- *)

datatype pattern =
  (* Primitive logical operations *)
    Boolean of bool
  | Not of pattern
  | And of pattern * pattern
  | Or of pattern * pattern
  | Implies of pattern * pattern
  | Iff of pattern * pattern

  (* Integer patterns *)
  | LessThan of integer * integer
  | LessEqual of integer * integer
  | IntegerEqual of integer * integer
  | GreaterEqual of integer * integer
  | GreaterThan of integer * integer

  (* Side patterns *)
  | SideEqual of side * side

  (* Block patterns *)
  | ConnectedBlock of Point.point * Point.point

  (* Edge patterns *)
  | Edge of edge * int

val compare : pattern * pattern -> order

val isValid : Dimensions.dimensions -> pattern -> bool

val transform : Symmetry.symmetry -> pattern -> pattern

val alwaysTrue : pattern

val alwaysFalse : pattern

val pp : pattern Print.pp

val toString : pattern -> string

(* ------------------------------------------------------------------------- *)
(* Booleans.                                                                 *)
(* ------------------------------------------------------------------------- *)

val destBoolean : pattern -> bool option

val isBoolean : pattern -> bool

val equalBoolean : bool -> pattern -> bool

val isTrue : pattern -> bool

val isFalse : pattern -> bool

(* ------------------------------------------------------------------------- *)
(* Side to move.                                                             *)
(* ------------------------------------------------------------------------- *)

val blackToMove : pattern

val whiteToMove : pattern

val mkSideToMove : Side.side -> pattern

val destSideToMove : pattern -> Side.side option

val isSideToMove : pattern -> bool

val equalSideToMove : Side.side -> pattern -> bool

(* ------------------------------------------------------------------------- *)
(* Individual points.                                                        *)
(* ------------------------------------------------------------------------- *)

val mkBlackStone : Point.point -> pattern

val mkWhiteStone : Point.point -> pattern

val mkSideStone : Side.side -> Point.point -> pattern

val mkStone : Point.point -> pattern

val mkEmpty : Point.point -> pattern

(* ------------------------------------------------------------------------- *)
(* Normalized patterns.                                                      *)
(* ------------------------------------------------------------------------- *)

type normalizedPattern = pattern

val normalize : pattern -> normalizedPattern

val isNormalized : pattern -> bool

val andNormalized : normalizedPattern -> normalizedPattern -> normalizedPattern

(* `strongerNormalized a b` iff `a ==> b` *)
val strongerNormalized : normalizedPattern -> normalizedPattern -> bool

(* `strictlyStrongerNormalized a b` iff `a ==> b /\ ~(b ==> a)` *)
val strictlyStrongerNormalized : normalizedPattern -> normalizedPattern -> bool

(* stronger is GREATER *)
val compareNormalized : normalizedPattern * normalizedPattern -> order option

end
