(* ========================================================================= *)
(* POINTS ON A GO BOARD                                                      *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Point :> Point =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* Helper functions                                                          *)
(* ------------------------------------------------------------------------- *)

fun string_to_char exn s = case String.explode s of [c] => c | _ => raise exn;

fun int_to_char exn n = string_to_char exn (Int.toString n);

(* ------------------------------------------------------------------------- *)
(* Files                                                                     *)
(* ------------------------------------------------------------------------- *)

val FILE_CHARS = String.explode "ABCDEFGHJKLMNOPQRSTUVWXYZ";

fun fileToChar n = List.nth (FILE_CHARS,n);

fun fileToString f = str (fileToChar f);

fun charToFile c =
    case index (equal (Char.toUpper c)) FILE_CHARS of
      NONE => raise Error "Point.charToFile: bad char"
    | SOME n => n;

(* ------------------------------------------------------------------------- *)
(* Ranks                                                                     *)
(* ------------------------------------------------------------------------- *)

fun rankToString n = Int.toString (n + 1);

fun stringToRank s =
    case Int.fromString s of
      NONE => raise Error "Point.stringToRank: not an integer"
    | SOME n =>
      if 1 <= n then n - 1
      else raise Error "Point.stringToRank: nonpositive integer";

(* ------------------------------------------------------------------------- *)
(* A type of points.                                                         *)
(* ------------------------------------------------------------------------- *)

datatype point = Point of {file : int, rank : int};

fun mk (x,y) = Point {file = x, rank = y};

fun compare (Point {file = f1, rank = r1}, Point {file = f2, rank = r2}) =
    case Int.compare (f1,f2) of
      LESS => LESS
    | EQUAL => Int.compare (r1,r2)
    | GREATER => GREATER;

fun equal (p1 : point) p2 = p1 = p2;

fun file (Point {file,...}) = file;

fun rank (Point {rank,...}) = rank;

(* ------------------------------------------------------------------------- *)
(* Parsing and pretty printing.                                              *)
(* ------------------------------------------------------------------------- *)

fun toString (Point {file,rank}) = fileToString file ^ rankToString rank;

val pp = Print.ppMap toString Print.ppString;

fun fromString s =
    case String.explode s of
      f :: r =>
      Point {file = charToFile f, rank = stringToRank (String.implode r)}
    | [] => raise Error "Point.fromString";

end

structure PointOrdered =
struct type t = Point.point val compare = Point.compare end

structure PointMap = KeyMap (PointOrdered);

structure PointSet =
struct

local
  structure S = ElementSet (PointMap);
in
  open S;
end;

fun mkRectangle (p1,p2) =
    if Point.equal p1 p2 then singleton p1
    else
      let
        val Point.Point {file = f1, rank = r1} = p1
        and Point.Point {file = f2, rank = r2} = p2

        val (f1,f2) = if f1 <= f2 then (f1,f2) else (f2,f1)
        and (r1,r2) = if r1 <= r2 then (r1,r2) else (r2,r1)

        fun addPoint file rank set =
            let
              val set = add set (Point.Point {file = file, rank = rank})
            in
              if rank = r2 then set else addPoint file (rank + 1) set
            end

        fun addColumn file set =
            let
              val set = addPoint file r1 set
            in
              if file = f2 then set else addColumn (file + 1) set
            end
      in
        addColumn f1 empty
      end;

val pp =
    Print.ppMap
      toList
      (Print.ppBracket "{" "}" (Print.ppOpList "," Point.pp));

end
