(* ========================================================================= *)
(* GO POSITIONS                                                              *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature Position =
sig

(* ------------------------------------------------------------------------- *)
(* A type of go position parameters.                                         *)
(* ------------------------------------------------------------------------- *)

type parameters =
    {rules : Rules.rules,
     komi : Komi.komi,
     dimensions : Dimensions.dimensions}

type squareParameters =
    {rules : Rules.rules,
     komi : Komi.komi,
     boardsize : int}

val equalParameters : parameters -> parameters -> bool

val default : parameters

val square : squareParameters -> parameters

(* ------------------------------------------------------------------------- *)
(* A type of go positions.                                                   *)
(* ------------------------------------------------------------------------- *)

type position

val new :
    {rules : Rules.rules,
     komi : Komi.komi,
     board : Board.board,
     toMove : Side.side,
     prisoners : int Side.sides} -> position

val parameters : position -> parameters

val initial : parameters -> position

val initialSquare : squareParameters -> position

val initialDefault : position

val isInitial : position -> bool

val rules : position -> Rules.rules

val komi : position -> Komi.komi

val board : position -> Board.board

val dimensions : position -> Dimensions.dimensions

val toMove : position -> Side.side

val prisoners : position -> int Side.sides

(* ------------------------------------------------------------------------- *)
(* Legal moves.                                                              *)
(* ------------------------------------------------------------------------- *)

val playMove : position -> Move.move -> position

val playMoves : position -> Move.move list -> position

val undo : position -> (position * Move.move) option

val history : position -> position * Move.move list

(* ------------------------------------------------------------------------- *)
(* Smart Game Format (SGF).                                                  *)
(* ------------------------------------------------------------------------- *)

val sgfNode : position -> GoSgf.node -> position

val sgfNodes : position -> GoSgf.node list -> position

val sgfMainVariation : position -> GoSgf.game -> position

val sgfFoldVariations :
    position -> (position * 'a list -> 'a) -> GoSgf.game -> 'a

val sgfGame : GoSgf.game -> position

val fromSgf : {filename : string} -> position  (* reads the first game *)

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

val pp : position Print.pp

val toString : position -> string

end
