(* ========================================================================= *)
(* GO POSITIONS                                                              *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Position :> Position =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* A type of go position parameters.                                         *)
(* ------------------------------------------------------------------------- *)

type parameters =
    {rules : Rules.rules,
     komi : Komi.komi,
     dimensions : Dimensions.dimensions};

type squareParameters =
    {rules : Rules.rules,
     komi : Komi.komi,
     boardsize : int};

val default =
    {rules = Rules.Chinese,
     komi = 7.5,
     dimensions = Dimensions.default};

fun equalParameters parm1 parm2 =
    let
      val {rules = rules1, komi = komi1, dimensions = dim1} = parm1
      and {rules = rules2, komi = komi2, dimensions = dim2} = parm2
    in
      rules1 = rules2 andalso
      Komi.equal komi1 komi2 andalso
      dim1 = dim2
    end;

fun square {rules,komi,boardsize} =
    let
      val dimensions = Dimensions.mkSquare boardsize
    in
      {rules = rules, komi = komi, dimensions = dimensions}
    end;

(* ------------------------------------------------------------------------- *)
(* A type of go positions.                                                   *)
(* ------------------------------------------------------------------------- *)

datatype position =
    Position of
      {fixed :
         {rules : Rules.rules,
          komi : Komi.komi},
       board : Board.board,
       toMove : Side.side,
       prisoners : int Side.sides,
       history : BoardSet.set,
       undo : (position * Move.move) option};

fun new {rules,komi,board,toMove,prisoners} =
    Position
      {fixed = {rules = rules, komi = komi},
       board = board,
       toMove = toMove,
       prisoners = prisoners,
       history = BoardSet.empty,
       undo = NONE};

fun parameters position =
    let
      val Position {fixed,board,...} = position
      val {rules,komi} = fixed
      val dimensions = Board.dimensions board
    in
      {rules = rules, komi = komi, dimensions = dimensions}
    end;

fun initial {rules,komi,dimensions} =
    new
      {rules = rules,
       komi = komi,
       board = Board.empty dimensions,
       toMove = Side.Black,
       prisoners = {black = 0, white = 0}};

fun initialSquare sq = initial (square sq);

val initialDefault = initial default;

fun isInitial position =
    let
      val Position {fixed = _, board, toMove, prisoners, history, undo} =
          position
    in
      Board.countStones board = {black = 0, white = 0} andalso
      Side.equal toMove Side.Black andalso
      prisoners = {black = 0, white = 0} andalso
      BoardSet.null history andalso
      not (Option.isSome undo)
    end;

fun rules (Position {fixed = {rules = r, ...}, ...}) = r;

fun komi (Position {fixed = {komi = k, ...}, ...}) = k;

fun board (Position {board = b, ...}) = b;

fun dimensions position = Board.dimensions (board position);

fun toMove (Position {toMove = s, ...}) = s;

fun prisoners (Position {prisoners = p, ...}) = p;

(* ------------------------------------------------------------------------- *)
(* Legal moves.                                                              *)
(* ------------------------------------------------------------------------- *)

fun updatePrisoners prisoners oldStones toMove newStones =
    let
      val {black = black_prisoners, white = white_prisoners} = prisoners
      and {black = black_old, white = white_old} = oldStones
      and {black = black_new, white = white_new} = newStones
      val black_prisoners = black_prisoners + black_old - black_new
      and white_prisoners = white_prisoners + white_old - white_new
    in
      case toMove of
        Side.Black => {black = black_prisoners + 1, white = white_prisoners}
      | Side.White => {black = black_prisoners, white = white_prisoners + 1}
    end;

fun playMove position Move.Pass =
    let
      val Position {fixed,board,toMove,prisoners,history,...} = position
      val toMove = Side.opponent toMove
      val undo = SOME (position,Move.Pass)
    in
      Position
        {fixed = fixed,
         board = board,
         toMove = toMove,
         prisoners = prisoners,
         history = history,
         undo = undo}
    end
  | playMove position (mv as Move.Stone point) =
    let
      val Position {fixed,board,toMove,prisoners,history,...} = position
      val {rules,...} = fixed
      val oldStones = Board.countStones board

      val history = BoardSet.add history board

      val board =
          let
            val permitSuicide = {permitSuicide = Rules.permitSuicide rules}
          in
            Board.placeStone permitSuicide board toMove point
          end

      val _ = not (BoardSet.member board history) orelse
              raise Error "forbidden by positional superko"

      val prisoners =
          updatePrisoners prisoners oldStones toMove (Board.countStones board)

      val toMove = Side.opponent toMove

      val undo = SOME (position,mv)
    in
      Position
        {fixed = fixed,
         board = board,
         toMove = toMove,
         prisoners = prisoners,
         history = history,
         undo = undo}
    end
    handle Error err => raise Error ("Position.move: " ^ err);

fun playMoves position [] = position
  | playMoves position (m :: ms) = playMoves (playMove position m) ms;

fun undo (Position {undo = u, ...}) = u;

val history =
    let
      fun f l p = case undo p of NONE => (p,l) | SOME (p,m) => f (m :: l) p
    in
      f []
    end;

(* ------------------------------------------------------------------------- *)
(* Smart Game Format (SGF).                                                  *)
(* ------------------------------------------------------------------------- *)

fun sgfPlayMove position side move =
    let
(*GomiTrace5
      val _ = Print.trace Side.pp "Position.sgfMove: side" side
      val _ = Print.trace Move.pp "Position.sgfMove: move" move
*)
      val position =
          if Side.equal side (toMove position) then position
          else playMove position Move.Pass
    in
      case move of
        Move.Pass => playMove position Move.Pass
      | Move.Stone point =>
        let
          val Position {fixed,board,prisoners,history,...} = position
          val oldStones = Board.countStones board

          val history = BoardSet.add history board

          val board =
              let
                val permitSuicide = {permitSuicide = true}
                val board = Board.clearPoints board (PointSet.singleton point)
              in
                Board.placeStone permitSuicide board side point
              end

          val prisoners =
              updatePrisoners
                prisoners oldStones side (Board.countStones board)

          val toMove = Side.opponent side

          val undo = SOME (position,move)
        in
          Position
            {fixed = fixed,
             board = board,
             toMove = toMove,
             prisoners = prisoners,
             history = history,
             undo = undo}
        end
    end;

fun sgfNode position (Sgf.Node props) =
    let
      fun exists ps = List.exists (fn p => StringMap.inDomain p props) ps
    in
      case (exists GoSgf.MOVE_PROPERTIES,
            exists GoSgf.SETUP_PROPERTIES) of
        (false,false) => position
      | (true,false) =>
        let
          val dim = dimensions position
        in
          case (GoSgf.peekB dim props, GoSgf.peekW dim props) of
            (NONE,NONE) => position
          | (SOME m, NONE) => sgfPlayMove position Side.Black m
          | (NONE, SOME m) => sgfPlayMove position Side.White m
          | (SOME _, SOME _) => raise Error "can't mix B and W properties"
        end
      | (false,true) =>
        let
          val Position {fixed,board,toMove,prisoners,...} = position

          val (board,dim) =
              case GoSgf.peekSZ props of
                SOME dim => (Board.empty dim, dim)
              | NONE => (board, Board.dimensions board)

          val black = Option.getOpt (GoSgf.peekAB dim props, PointSet.empty)
          and white = Option.getOpt (GoSgf.peekAW dim props, PointSet.empty)
          and empty = Option.getOpt (GoSgf.peekAE dim props, PointSet.empty)

(*GomiTrace5
          val _ = Print.trace PointSet.pp "Position.sgfNode: black" black
          val _ = Print.trace PointSet.pp "Position.sgfNode: white" white
          val _ = Print.trace PointSet.pp "Position.sgfNode: empty" empty
*)

          val board =
              let
                fun place side (p,b) =
                    Board.placeStone {permitSuicide = false} b side p

                val pts = PointSet.unionList [black,white,empty]
                val _ = PointSet.disjoint black white orelse
                        raise Error "AB and AW properties overlap"
                val _ = PointSet.disjoint black empty orelse
                        raise Error "AB and AE properties overlap"
                val _ = PointSet.disjoint white empty orelse
                        raise Error "AW and AE properties overlap"

                val board = Board.clearPoints board pts
                val board = PointSet.foldl (place Side.Black) board black
                val board = PointSet.foldl (place Side.White) board white
              in
                board
              end

          val toMove = Option.getOpt (GoSgf.peekPL props, toMove)
        in
          Position
            {fixed = fixed,
             board = board,
             toMove = toMove,
             prisoners = prisoners,
             history = BoardSet.empty,
             undo = NONE}
        end
      | (true,true) => raise Error "can't mix move and setup properties"
    end
    handle Error err => raise Error ("Position.sgfNode: " ^ err);

local
  fun applyNode (n,p) = sgfNode p n;
in
  fun sgfNodes position nodes = List.foldl applyNode position nodes;
end;

fun sgfMainVariation position (Sgf.Game (nodes,subgames)) =
    let
      val position = sgfNodes position nodes
    in
      case subgames of
        [] => position
      | mainVariation :: _ => sgfMainVariation position mainVariation
    end;

fun sgfFoldVariations position f (Sgf.Game (nodes,subgames)) =
    let
      val position = sgfNodes position nodes
      val l = List.map (sgfFoldVariations position f) subgames
    in
      f (position,l)
    end;

val sgfGame = sgfMainVariation initialDefault;

fun fromSgf filename =
    case GoSgf.read filename of
      [] => raise Bug "Position.fromSgf: empty collection"
    | game :: _ => sgfGame game;

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

val pp = Print.ppMap board Board.pp;

val toString = Print.toString pp;

end
