(* ========================================================================= *)
(* ESTIMATING PROBABILITIES                                                  *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature Probability =
sig

(* ------------------------------------------------------------------------- *)
(* A type of probability estimates.                                          *)
(* ------------------------------------------------------------------------- *)

type probability

val complement : probability -> probability

val conj : probability -> probability -> probability

val disj : probability -> probability -> probability

val max : probability -> probability -> probability

val min : probability -> probability -> probability

val combine : probability -> probability -> probability

(* ------------------------------------------------------------------------- *)
(* Statistics.                                                               *)
(* ------------------------------------------------------------------------- *)

val pdf : probability -> real -> real

val cdf : probability -> real -> real

val expectation : probability -> (real -> real) -> real

val mean : probability -> real

val variance : probability -> real

val standardDeviation : probability -> real

val median : probability -> real

val lowerQuartile : probability -> real

val upperQuartile : probability -> real

val quartile : probability -> real -> real

val lessThan : probability -> probability -> real

val maxLessThan : probability -> probability -> real

(* ------------------------------------------------------------------------- *)
(* Distances between probability estimate cdfs.                              *)
(* ------------------------------------------------------------------------- *)

val distanceLp : real -> probability -> probability -> real

val distanceL1 : probability -> probability -> real

val distanceL2 : probability -> probability -> real

val distanceLinf : probability -> probability -> real

(* ------------------------------------------------------------------------- *)
(* Distributions over probabilities.                                         *)
(* ------------------------------------------------------------------------- *)

val uniformDistribution : probability

val betaDistribution : {alpha : real, beta : real} -> probability

(* ------------------------------------------------------------------------- *)
(* Estimating probabilities from frequency counts.                           *)
(* ------------------------------------------------------------------------- *)

type frequency = {count : int, total : int}

val zeroFrequency : frequency

val fromFrequency : frequency -> probability

(* ------------------------------------------------------------------------- *)
(* A (multi-)set of probability estimates.                                   *)
(* ------------------------------------------------------------------------- *)

type set = probability IntMap.map

val emptySet : set

val addSet : set -> int * probability -> set

val isMaxSet : set -> real IntMap.map

val randomMaxSet : set -> int

(* ------------------------------------------------------------------------- *)
(* Parsing and pretty printing.                                              *)
(* ------------------------------------------------------------------------- *)

val inverseConjInfo : string

val pp : probability Print.pp

val toString : probability -> string

end
