(* ========================================================================= *)
(* GO RULES                                                                  *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Rules :> Rules =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* A type of Go rules.                                                       *)
(*                                                                           *)
(* The different rulesets are compared at                                    *)
(*                                                                           *)
(*   http://www.britgo.org/rules/compare.html                                *)
(* ------------------------------------------------------------------------- *)

datatype rules =
    American
  | Chinese
  | Japanese
  | NewZealand
  | SST;

val allRules =
    [American,
     Chinese,
     Japanese,
     NewZealand,
     SST];

(* ------------------------------------------------------------------------- *)
(* Suicide.                                                                  *)
(* ------------------------------------------------------------------------- *)

fun permitSuicide rules =
    case rules of
      American => false
    | Chinese => false
    | Japanese => false
    | NewZealand => true
    | SST => true;

(* ------------------------------------------------------------------------- *)
(* Komi.                                                                     *)
(* ------------------------------------------------------------------------- *)

fun komi rules {handicap = 0} =
    (case rules of
       American => 5.5
     | Chinese => 7.5
     | Japanese => 6.5
     | NewZealand => 7.0
     | SST => 7.5)
  | komi rules {handicap = n} =
    (case rules of
       American => Real.fromInt (n - 1)
     | Chinese => Real.fromInt n
     | Japanese => 0.0
     | NewZealand => 0.0
     | SST => 0.0);

(* ------------------------------------------------------------------------- *)
(* Parsing and pretty printing.                                              *)
(* ------------------------------------------------------------------------- *)

fun toString rules =
    case rules of
      American => "American"
    | Chinese => "Chinese"
    | Japanese => "Japanese"
    | NewZealand => "NewZealand"
    | SST => "SST";

local
  fun trans c = if Char.isAlphaNum c then str (Char.toUpper c) else "";
in
  fun fromString s =
      case String.translate trans s of
        "AMERICAN" => American
      | "CHINESE" => Chinese
      | "JAPANESE" => Japanese
      | "NEWZEALAND" => NewZealand
      | "SST" => SST
      | _ => raise Error ("Unknown ruleset \"" ^ s ^ "\"");
end;

end
