(* ========================================================================= *)
(* GO SCORES                                                                 *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Score :> Score =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* A type of go scores.                                                      *)
(* ------------------------------------------------------------------------- *)

datatype score = Win of Side.side * real | Draw;

fun isWin Draw _ = false
  | isWin (Win (side,_)) side' = side = side';

fun fromPoints komi =
    let
      val minBlack = Komi.minPointsBlackWin komi
      and maxWhite = Komi.maxPointsWhiteWin komi
    in
      fn points =>
         if points >= minBlack then
           Win (Side.Black, Real.fromInt points - komi)
         else if points <= maxWhite then
           Win (Side.White, komi - Real.fromInt points)
         else
           Draw
    end;

(* ------------------------------------------------------------------------- *)
(* Parsing and pretty printing.                                              *)
(* ------------------------------------------------------------------------- *)

fun plusToString x =
    let
      val n = Real.round (100.0 * x)
    in
      if n mod 10 <> 0 then Real.fmt (StringCvt.FIX (SOME 2)) x
      else if n mod 100 <> 0 then Real.fmt (StringCvt.FIX (SOME 1)) x
      else Int.toString n
    end;

fun toString (Win (s,r)) = Side.toString s ^ "+" ^ plusToString r
  | toString Draw = "0";

fun fromString "0" = Draw
  | fromString s =
    (case String.fields (equal #"+") s of
       [s,r] =>
       (case Real.fromString r of
          NONE => raise Error "couldn't parse number"
        | SOME r =>
          if r > 0.0 then Win (Side.fromString s, r)
          else raise Error "nonpositive number")
     | [_] => raise Error "no plus symbol"
     | _ => raise Error "multiple plus symbols")
    handle Error err => raise Error ("Score.fromString: " ^ err);

end
