(* ========================================================================= *)
(* THE SMART GAME FILE FORMAT (SGF v4)                                       *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature Sgf =
sig

(* ------------------------------------------------------------------------- *)
(* SGF properties.                                                           *)
(* ------------------------------------------------------------------------- *)

type propertyIdentifier = string

datatype ('point,'move,'stone) propertyValue =
    None
  | Number of int
  | Real of real
  | Side of Side.side
  | Text of string
  | Point of 'point
  | Move of 'move
  | Stone of 'stone
  | Compose of ('point,'move,'stone) propertyValue *
               ('point,'move,'stone) propertyValue
  | Unknown of string  (* completely untouched *)

type ('point,'move,'stone) properties =
     ('point,'move,'stone) propertyValue list StringMap.map

type 'a rectangle = 'a * 'a

val MOVE_PROPERTIES : propertyIdentifier list

val SETUP_PROPERTIES : propertyIdentifier list

val peekAB : ('point,'move,'stone) properties -> 'stone rectangle list option

val peekAE : ('point,'move,'stone) properties -> 'point rectangle list option

val peekAW : ('point,'move,'stone) properties -> 'stone rectangle list option

val peekB : ('point,'move,'stone) properties -> 'move option

val peekPL : ('point,'move,'stone) properties -> Side.side option

val peekSZ : ('point,'move,'stone) properties -> int rectangle option

val peekW : ('point,'move,'stone) properties -> 'move option

(* ------------------------------------------------------------------------- *)
(* SGF types.                                                                *)
(* ------------------------------------------------------------------------- *)

datatype ('point,'move,'stone) node =
    Node of ('point,'move,'stone) properties

datatype ('point,'move,'stone) game =
    Game of ('point,'move,'stone) node list * ('point,'move,'stone) game list

type ('point,'move,'stone) collection = ('point,'move,'stone) game list

(* ------------------------------------------------------------------------- *)
(* Pretty-printing.                                                          *)
(* ------------------------------------------------------------------------- *)

val ppPropertyValue :
    {ppPoint : 'point Print.pp,
     ppMove : 'move Print.pp,
     ppStone : 'stone Print.pp} ->
    ('point,'move,'stone) propertyValue Print.pp

val ppNode :
    {ppPoint : 'point Print.pp,
     ppMove : 'move Print.pp,
     ppStone : 'stone Print.pp} ->
    ('point,'move,'stone) node Print.pp

val ppGame :
    {ppPoint : 'point Print.pp,
     ppMove : 'move Print.pp,
     ppStone : 'stone Print.pp} ->
    ('point,'move,'stone) game Print.pp

(* ------------------------------------------------------------------------- *)
(* Parsing.                                                                  *)
(* ------------------------------------------------------------------------- *)

type 'a parser = (char,'a) Parse.parser

type ('point,'move,'stone) propertyParser =
     propertyIdentifier -> ('point,'move,'stone) propertyValue list parser

val noneParser : ('point,'move,'stone) propertyValue parser

val numberParser : ('point,'move,'stone) propertyValue parser

val realParser : ('point,'move,'stone) propertyValue parser

val sideParser : ('point,'move,'stone) propertyValue parser

val textParser :
    {stopOnCompose : bool} -> ('point,'move,'stone) propertyValue parser

val pointParser : 'point parser -> ('point,'move,'stone) propertyValue parser

val moveParser : 'move parser -> ('point,'move,'stone) propertyValue parser

val stoneParser : 'stone parser -> ('point,'move,'stone) propertyValue parser

val doubleParser : ('point,'move,'stone) propertyValue parser

val simpleTextParser :
    {stopOnCompose : bool} -> ('point,'move,'stone) propertyValue parser

val emptyParser : ('point,'move,'stone) propertyValue list parser

val singleParser :
    ('point,'move,'stone) propertyValue parser ->
    ('point,'move,'stone) propertyValue list parser

val listParser :
    ('point,'move,'stone) propertyValue parser ->
    ('point,'move,'stone) propertyValue list parser

val eListParser :
    ('point,'move,'stone) propertyValue parser ->
    ('point,'move,'stone) propertyValue list parser

val pointListParser :
    'point parser -> ('point,'move,'stone) propertyValue list parser

val pointEListParser :
    'point parser -> ('point,'move,'stone) propertyValue list parser

val stoneListParser :
    'stone parser -> ('point,'move,'stone) propertyValue list parser

val stoneEListParser :
    'stone parser -> ('point,'move,'stone) propertyValue list parser

val generalPropertyParser :
    {pointParser : 'point parser,
     moveParser : 'move parser,
     stoneParser : 'stone parser} ->
    ('point,'move,'stone) propertyParser

val gameParser :
    {propertyParser : ('point,'move,'stone) propertyParser} ->
    ('point,'move,'stone) game parser

(* ------------------------------------------------------------------------- *)
(* Reading from a file.                                                      *)
(* ------------------------------------------------------------------------- *)

val read :
    {filename : string,
     propertyParser : ('point,'move,'stone) propertyParser} ->
    ('point,'move,'stone) collection

(* ------------------------------------------------------------------------- *)
(* Writing to a file.                                                        *)
(* ------------------------------------------------------------------------- *)

val write :
    {ppPoint : 'point Print.pp,
     ppMove : 'move Print.pp,
     ppStone : 'stone Print.pp,
     collection : ('point,'move,'stone) collection,
     filename : string} -> unit

end
