(* ========================================================================= *)
(* THE SMART GAME FILE FORMAT (SGF v4)                                       *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Sgf :> Sgf =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* SGF properties.                                                           *)
(* ------------------------------------------------------------------------- *)

type propertyIdentifier = string;

datatype ('point,'move,'stone) propertyValue =
    None
  | Number of int
  | Real of real
  | Side of Side.side
  | Text of string
  | Point of 'point
  | Move of 'move
  | Stone of 'stone
  | Compose of ('point,'move,'stone) propertyValue *
               ('point,'move,'stone) propertyValue
  | Unknown of string;

type ('point,'move,'stone) properties =
     ('point,'move,'stone) propertyValue list StringMap.map;

type 'a rectangle = 'a * 'a;

val MOVE_PROPERTIES =
    ["B","BL","BM","DO","IT","KO","MN","OB","OW","TE","W","WL"];

val SETUP_PROPERTIES =
    ["AB","AE","AW","PL"];

fun propertiesFromList propl =
    let
      val props = StringMap.fromList propl
      val _ = length propl = StringMap.size props orelse
              raise Error "duplicate property identifiers"
    in
      props
    end;

local
  fun destList dest =
      let
        fun f acc [] = SOME (List.rev acc)
          | f acc (p :: pl) =
            case dest p of
              NONE => NONE
            | SOME x => f (x :: acc) pl
      in
        f []
      end;

  fun destRectangle dest (Compose (x,y)) =
      (case dest x of
         NONE => NONE
       | SOME x =>
         (case dest y of
            NONE => NONE
          | SOME y => SOME (x,y)))
    | destRectangle dest x =
      case dest x of
        NONE => NONE
      | SOME x => SOME (x,x);

  fun destNumber (Number n) = SOME n
    | destNumber _ = NONE;

  fun destPoint (Point point) = SOME point
    | destPoint _ = NONE;

  fun destPointList l = destList (destRectangle destPoint) l;

  fun destStone (Stone stone) = SOME stone
    | destStone _ = NONE;

  fun destStoneList l = destList (destRectangle destStone) l;

  fun peekSide props propId =
      case StringMap.peek props propId of
        SOME [Side s] => SOME s
      | SOME _ => raise Bug "bad value"
      | _ => NONE;

  fun peekNumberRectangle props propId =
      case StringMap.peek props propId of
        SOME [x] =>
        (case destRectangle destNumber x of
           NONE => raise Bug "bad value"
         | s => s)
      | SOME _ => raise Bug "bad values"
      | _ => NONE;

  fun peekStoneList props propId =
      case StringMap.peek props propId of
        NONE => NONE
      | SOME pts =>
        case destStoneList pts of
          NONE => raise Bug "bad value"
        | SOME [] => raise Bug "empty list of stones"
        | s => s;

  fun peekPointList props propId =
      case StringMap.peek props propId of
        NONE => NONE
      | SOME pts =>
        case destPointList pts of
          NONE => raise Bug "bad value"
        | SOME [] => raise Bug "empty list of points"
        | s => s;

  fun peekStoneList props propId =
      case StringMap.peek props propId of
        NONE => NONE
      | SOME pts =>
        case destStoneList pts of
          NONE => raise Bug "bad value"
        | SOME [] => raise Bug "empty list of stones"
        | s => s;

  fun peekMove props propId =
      case StringMap.peek props propId of
        SOME [Move m] => SOME m
      | SOME _ => raise Bug "bad value"
      | _ => NONE;
in
  fun peekAB props =
      peekStoneList props "AB"
      handle Bug bug => raise Bug ("Sgf.peekAB: " ^ bug);

  fun peekAE props =
      peekPointList props "AE"
      handle Bug bug => raise Bug ("Sgf.peekAE: " ^ bug);

  fun peekAW props =
      peekStoneList props "AW"
      handle Bug bug => raise Bug ("Sgf.peekAW: " ^ bug);

  fun peekB props =
      peekMove props "B"
      handle Bug bug => raise Bug ("Sgf.peekB: " ^ bug);

  fun peekPL props =
      peekSide props "PL"
      handle Bug bug => raise Bug ("Sgf.peekPL: " ^ bug);

  fun peekSZ props =
      peekNumberRectangle props "SZ"
      handle Bug bug => raise Bug ("Sgf.peekSZ: " ^ bug);

  fun peekW props =
      peekMove props "W"
      handle Bug bug => raise Bug ("Sgf.peekW: " ^ bug);
end;

(* ------------------------------------------------------------------------- *)
(* SGF types.                                                                *)
(* ------------------------------------------------------------------------- *)

datatype ('point,'move,'stone) node =
    Node of ('point,'move,'stone) properties;

datatype ('point,'move,'stone) game =
    Game of ('point,'move,'stone) node list * ('point,'move,'stone) game list;

type ('point,'move,'stone) collection = ('point,'move,'stone) game list;

(* ------------------------------------------------------------------------- *)
(* Pretty-printing.                                                          *)
(* ------------------------------------------------------------------------- *)

local
  val escapeChars = "\\]";
in
  fun escapeText {escapeCompose} =
      let
        fun escape #":" = escapeCompose
          | escape c = Char.contains escapeChars c

        fun trans c = if escape c then "\\" ^ str c else str c
      in
        String.translate trans
      end;
end;

fun breakBefore ppX x = Print.sequence (Print.breaks 0) (ppX x);

fun ppPropertyValue {ppPoint,ppMove,ppStone} prop =
    let
      fun ppProp _ None = Print.skip
        | ppProp _ (Number n) = Print.ppInt n
        | ppProp _ (Real r) =
          Print.ppString (Real.fmt (StringCvt.FIX (SOME 3)) r)
        | ppProp _ (Side s) = Side.pp s
        | ppProp esc (Text s) = Print.ppString (escapeText esc s)
        | ppProp _ (Point pt) = ppPoint pt
        | ppProp _ (Move m) = ppMove m
        | ppProp _ (Stone s) = ppStone s
        | ppProp {escapeCompose} (Compose (x,y)) =
          Print.program
            [ppProp {escapeCompose = true} x,
             Print.ppString ":",
             ppProp {escapeCompose = escapeCompose} y]
        | ppProp _ (Unknown s) = Print.ppString s
    in
      Print.inconsistentBlock 0 [ppProp {escapeCompose = false} prop]
    end;

fun ppProperty info (propId,propVals) =
    let
      fun ppProp v =
          Print.program
            [Print.ppString "[",
             ppPropertyValue info v,
             Print.ppString "]"]
    in
      Print.inconsistentBlock 0
        [Print.ppString propId,
         case propVals of
           [] => ppProp None
         | v :: vs =>
           Print.program (ppProp v :: List.map (breakBefore ppProp) vs)]
    end;

fun ppNode info (Node prop) =
    let
      val ppP = ppProperty info
    in
      Print.inconsistentBlock 0
        [Print.ppString ";",
         case StringMap.toList prop of
           [] => Print.skip
         | x :: xs => Print.program (ppP x :: List.map (breakBefore ppP) xs)]
    end;

fun ppGame info game =
    let
      val ppN = ppNode info

      fun ppG (Game (nodes,subgames)) =
          Print.program
            [Print.ppString "(",
             case nodes of
               [] => Print.skip
             | n :: ns => Print.program (ppN n :: List.map (breakBefore ppN) ns),
             case subgames of
               [] => Print.skip
             | g :: gs => Print.program (ppG g :: List.map (breakBefore ppG) gs)]
    in
      Print.inconsistentBlock 0 [ppG game]
    end;

(* ------------------------------------------------------------------------- *)
(* Parsing.                                                                  *)
(* ------------------------------------------------------------------------- *)

type 'a parser = (char,'a) Parse.parser;

type ('point,'move,'stone) propertyParser =
     propertyIdentifier -> ('point,'move,'stone) propertyValue list parser;

local
  infixr 9 >>++
  infixr 8 ++
  infixr 7 >>
  infixr 6 ||

  open Parse;

  val space = many (some Char.isSpace) >> K ();

  fun spacedR p = p ++ space >> fst;

  fun spacedLR p = space ++ p >> snd;

  val newlineParser =
      (exactChar #"\n" ++ optional (exactChar #"\r") ||
       exactChar #"\r" ++ optional (exactChar #"\n")) >> K ();

  val signParser =
      exactChar #"+" >> K 1 ||
      exactChar #"-" >> K (~1);

  fun possiblePairParser parser construct =
      parser >>++
      (fn x =>
          exactChar #":" ++ parser >>
          (fn (_,y) => Compose (construct x, construct y)) ||
          nothing >> K (construct x));

  fun propertyBracketsParser parser =
      exactChar #"[" ++ space ++ parser ++ space ++ exactChar #"]" ++ space >>
      (fn (_,((),(propVal,((),(_,()))))) => propVal);

  fun integerParser input =
      let
        fun mkInteger (s,i) =
            let
              val s = Option.getOpt (s,1)
              val s = if s < 0 then [#"~"] else []

              val x = String.implode (s @ i)
            in
              case Int.fromString x of
                SOME i => i
              | NONE => raise Bug ("integerParser: " ^ x)
            end
      in
        optional signParser ++ atLeastOne (some Char.isDigit) >> mkInteger
      end input;
in
  (* Primitive propertyValue parsers *)

  fun noneParser input = (nothing >> K None) input;

  fun numberParser input = (integerParser >> Number) input;

  fun realParser input =
      let
        fun mkReal (s,(i,f)) =
            let
              val s = Option.getOpt (s,1)
              val s = if s < 0 then [#"~"] else []

              val f = case f of SOME f => #"." :: f | NONE => []

              val x = String.implode (s @ i @ f)
            in
              case Real.fromString x of
                SOME r => Real r
              | NONE => raise Bug ("realParser: " ^ x)
            end
      in
        optional signParser ++ atLeastOne (some Char.isDigit) ++
        optional (exactChar #"." ++ atLeastOne (some Char.isDigit) >> snd) >>
        mkReal
      end input;

  fun sideParser input =
      (exactChar #"B" >> K (Side Side.Black) ||
       exactChar #"W" >> K (Side Side.White)) input;

  fun textParser {stopOnCompose} =
      let
        val spaceParser = some Char.isSpace >> K [#" "]

        val escapeParser =
            atLeastOne newlineParser >> K [] ||
            spaceParser ||
            any >> singleton

        fun stopOn #"]" = true
          | stopOn #":" = stopOnCompose
          | stopOn _ = false

        val lexParser =
            exactChar #"\\" ++ escapeParser >> snd ||
            newlineParser >> K [#"\n"] ||
            spaceParser ||
            some (not o stopOn) >> singleton;
      in
        many lexParser >> (fn l => Text (String.implode (List.concat l)))
      end;

  fun pointParser p = p >> Point

  fun moveParser m = m >> Move

  fun stoneParser s = s >> Stone

  fun composeParser parser1 parser2 =
      parser1 ++ exactChar #":" ++ parser2 >> (fn (x,(_,y)) => Compose (x,y));

  fun doubleParser input =
      (exactChar #"1" >> K (Number 1) ||
       exactChar #"2" >> K (Number 2)) input;

  fun simpleTextParser {stopOnCompose} =
      let
        val escapeParser =
            atLeastOne newlineParser >> K [] ||
            some Char.isSpace >> K [#" "] ||
            any >> singleton

        fun stopOn #"]" = true
          | stopOn #":" = stopOnCompose
          | stopOn _ = false

        val lexParser =
            exactChar #"\\" ++ escapeParser >> snd ||
            atLeastOne (some Char.isSpace) >> K [#" "] ||
            some (not o stopOn) >> singleton;
      in
        many lexParser >> (fn l => Text (String.implode (List.concat l)))
      end;

  fun unknownParser input =
      let
        fun stopOn #"]" = true
          | stopOn _ = false

        val lexParser =
            exactChar #"\\" ++ any >> (fn (_,c) => [#"\\",c]) ||
            some (fn c => not (stopOn c)) >> singleton;
      in
        many lexParser >> (fn l => Unknown (String.implode (List.concat l)))
      end input;

  (* Derived propertyValue parsers *)

  fun applicationParser input =
      composeParser
        (simpleTextParser {stopOnCompose = true})
        (simpleTextParser {stopOnCompose = true}) input;

  fun arrowParser p = composeParser (pointParser p) (pointParser p);

  fun figureParser input =
      composeParser
        numberParser (simpleTextParser {stopOnCompose = true}) input;

  fun labelParser p =
      composeParser
        (pointParser p) (simpleTextParser {stopOnCompose = true});

  fun lineParser p = composeParser (pointParser p) (pointParser p);

  fun sizeParser input = possiblePairParser integerParser Number input;

  (* propertyValue list parsers *)

  fun emptyParser input = (propertyBracketsParser noneParser >> K []) input;

  fun singleParser parser = propertyBracketsParser parser >> singleton;

  fun listParser parser = atLeastOne (propertyBracketsParser parser);

  fun eListParser parser = listParser parser || emptyParser;

  fun pointListParser pointParser =
      listParser (possiblePairParser pointParser Point);

  fun pointEListParser pointParser =
      pointListParser pointParser || emptyParser;

  fun stoneListParser stoneParser =
      listParser (possiblePairParser stoneParser Stone);

  fun stoneEListParser stoneParser =
      stoneListParser stoneParser || emptyParser;

  (* The game parser *)

  fun gameParser {propertyParser} =
      let
        val propIdentifierParser =
            atLeastOne (some Char.isUpper) ++ space >>
            (fn (cs,()) => String.implode cs)

        fun propValueParser propId input =
            let
(*GomiTrace1
              val () = trace ("gamePrint.propId: " ^ propId ^ "\n")
*)
            in
              propertyParser propId >> (fn propVal => (propId,propVal))
            end input

        val propParser = propIdentifierParser >>++ propValueParser

        val nodeParser =
            exactChar #";" ++ space ++ many propParser >>
            (fn (_,((),props)) => Node (propertiesFromList props))

        fun parser input =
            (space ++ exactChar #"(" ++ space ++ atLeastOne nodeParser ++
             many parser ++ exactChar #")" ++ space >>
             (fn ((),(_,((),(nodes,(subgames,(_,())))))) =>
                 Game (nodes,subgames))) input
      in
        parser
      end;

  (* The top level SGF collection parser *)

  fun parseSgf propParser chars =
      let
        val games = Parse.everything (gameParser propParser >> singleton) chars
        val _ = not (Stream.null games) orelse raise Error "parseSgf: no games"
      in
        games
      end;
end;

fun generalPropertyParser info =
    let
      val {pointParser = p, moveParser = m, stoneParser = s} = info
    in
      (* Properties with type "move" *)
      fn "B" => singleParser (moveParser m)
       | "BL" => singleParser realParser
       | "BM" => singleParser doubleParser
       | "DO" => emptyParser
       | "IT" => emptyParser
       | "KO" => emptyParser
       | "MN" => singleParser numberParser
       | "OB" => singleParser numberParser
       | "OW" => singleParser numberParser
       | "TE" => singleParser doubleParser
       | "W" => singleParser (moveParser m)
       | "WL" => singleParser realParser

      (* Properties with type "setup" *)
       | "AB" => stoneListParser s
       | "AE" => pointListParser p
       | "AW" => stoneListParser s
       | "PL" => singleParser sideParser

      (* Properties with type "-" *)
       | "AR" => listParser (arrowParser p)
       | "C" => singleParser (textParser {stopOnCompose = false})
       | "CR" => pointListParser p
       | "DM" => singleParser doubleParser
       | "FG" => singleParser figureParser
       | "GB" => singleParser doubleParser
       | "GW" => singleParser doubleParser
       | "HO" => singleParser doubleParser
       | "LB" => listParser (labelParser p)
       | "LN" => listParser (lineParser p)
       | "MA" => pointListParser p
       | "N" => singleParser (simpleTextParser {stopOnCompose = false})
       | "SL" => pointListParser p
       | "SQ" => pointListParser p
       | "TR" => pointListParser p
       | "UC" => singleParser doubleParser
       | "V" => singleParser realParser

      (* Properties with type "- (inherit)" *)
       | "DD" => pointEListParser p
       | "PM" => singleParser numberParser
       | "VW" => pointListParser p

      (* Properties with type "root" *)
       | "AP" => singleParser applicationParser
       | "CA" => singleParser (simpleTextParser {stopOnCompose = false})
       | "FF" => singleParser numberParser
       | "GM" => singleParser numberParser
       | "ST" => singleParser numberParser
       | "SZ" => singleParser sizeParser

      (* Properties with type "game-info" *)
       | "AN" => singleParser (simpleTextParser {stopOnCompose = false})
       | "BR" => singleParser (simpleTextParser {stopOnCompose = false})
       | "BT" => singleParser (simpleTextParser {stopOnCompose = false})
       | "CP" => singleParser (simpleTextParser {stopOnCompose = false})
       | "DT" => singleParser (simpleTextParser {stopOnCompose = false})
       | "EV" => singleParser (simpleTextParser {stopOnCompose = false})
       | "GC" => singleParser (textParser {stopOnCompose = false})
       | "GN" => singleParser (simpleTextParser {stopOnCompose = false})
       | "ON" => singleParser (simpleTextParser {stopOnCompose = false})
       | "OT" => singleParser (simpleTextParser {stopOnCompose = false})
       | "PB" => singleParser (simpleTextParser {stopOnCompose = false})
       | "PC" => singleParser (simpleTextParser {stopOnCompose = false})
       | "PW" => singleParser (simpleTextParser {stopOnCompose = false})
       | "RE" => singleParser (simpleTextParser {stopOnCompose = false})
       | "RO" => singleParser (simpleTextParser {stopOnCompose = false})
       | "RU" => singleParser (simpleTextParser {stopOnCompose = false})
       | "SO" => singleParser (simpleTextParser {stopOnCompose = false})
       | "TM" => singleParser realParser
       | "US" => singleParser (simpleTextParser {stopOnCompose = false})
       | "WR" => singleParser (simpleTextParser {stopOnCompose = false})
       | "WT" => singleParser (simpleTextParser {stopOnCompose = false})

      (* Handling unknown properties *)
       | _ => eListParser unknownParser
    end;

(* ------------------------------------------------------------------------- *)
(* Reading from a file.                                                      *)
(* ------------------------------------------------------------------------- *)

fun read {filename,propertyParser} =
    let
      val lines = Stream.fromTextFile {filename = filename}

      val chars =
          let
            fun f line = Stream.fromList (String.explode line)
          in
            Stream.concat (Stream.map f lines)
          end

      val games = parseSgf {propertyParser = propertyParser} chars
    in
      Stream.toList games
    end;

(* ------------------------------------------------------------------------- *)
(* Writing to a file.                                                        *)
(* ------------------------------------------------------------------------- *)

fun write {ppPoint,ppMove,ppStone,collection,filename} =
    let
      val gameToString =
          Print.toString
            (ppGame {ppPoint = ppPoint, ppMove = ppMove, ppStone = ppStone})

      fun gameStream [] () = Stream.Nil
        | gameStream (h :: t) () =
          Stream.Cons ("\n" ^ gameToString h, gameStream t)
    in
      case collection of
        [] => raise Error "Sgf.write: no games"
      | game :: games =>
        Stream.toTextFile
          {filename = filename}
          (Stream.Cons (gameToString game, gameStream games))
    end;

end
