(* ========================================================================= *)
(* THE STATUS OF A POINT ON A GO BOARD                                       *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Status :> Status =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* A type of point statuses.                                                 *)
(* ------------------------------------------------------------------------- *)

datatype status =
    Stone of Side.side
  | Eye of Side.side
  | Seki;

fun compare (Stone side1, Stone side2) = Side.compare (side1,side2)
  | compare (Stone _, _) = LESS
  | compare (_, Stone _) = GREATER
  | compare (Eye side1, Eye side2) = Side.compare (side1,side2)
  | compare (Eye _, _) = LESS
  | compare (_, Eye _) = GREATER
  | compare (Seki,Seki) = EQUAL;

val allStatuses =
    [Stone Side.Black, Stone Side.White,
     Eye Side.Black, Eye Side.White,
     Seki];

(*GomiDebug
val () = if sort compare allStatuses = allStatuses then ()
         else raise Bug "Status.allStatuses: out of order";
*)

fun fold f b = List.foldl f b allStatuses;

fun transform (Symmetry.Symmetry {opponent,...}) status =
    if not opponent then status
    else
      case status of
        Stone s => Stone (Side.opponent s)
      | Eye s => Eye (Side.opponent s)
      | Seki => status;

(* ------------------------------------------------------------------------- *)
(* Status sides.                                                             *)
(* ------------------------------------------------------------------------- *)

fun destStone (Stone s) = SOME s
  | destStone _ = NONE;

fun isStone s = Option.isSome (destStone s);

fun destEye (Eye s) = SOME s
  | destEye _ = NONE;

fun isEye s = Option.isSome (destEye s);

fun destTerritory (Stone s) = SOME s
  | destTerritory (Eye s) = SOME s
  | destTerritory _ = NONE;

fun isTerritory s = Option.isSome (destTerritory s);

(* ------------------------------------------------------------------------- *)
(* Pretty printing.                                                          *)
(* ------------------------------------------------------------------------- *)

fun toString (Stone side) = Side.toString side ^ " stone"
  | toString (Eye side) = Side.toString side ^ " eye"
  | toString Seki = "Seki";

val pp = Print.ppMap toString Print.ppString;

end

structure StatusOrdered =
struct type t = Status.status val compare = Status.compare end

structure StatusMap = KeyMap (StatusOrdered);

structure StatusSet =
struct

local
  structure S = ElementSet (StatusMap);
in
  open S;
end;

fun transform (sym as Symmetry.Symmetry {opponent,...}) set =
    if not opponent then set
    else
      let
        fun f (s,(id,acc)) =
            let
              val s' = Status.transform sym s
              val id = id andalso Portable.pointerEqual (s',s)
              val acc = add acc s'
            in
              (id,acc)
            end

        val (id,set') = foldl f (true,empty) set
      in
        if id then set else set'
      end;

val pp =
    Print.ppMap
      toList
      (Print.ppBracket "{" "}" (Print.ppOpList "," Status.pp));

end
