(* ========================================================================= *)
(* GO BOARD SYMMETRY                                                         *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

signature Symmetry =
sig

(* ------------------------------------------------------------------------- *)
(* The group of go board symmetries.                                         *)
(* ------------------------------------------------------------------------- *)

datatype symmetry =
    Symmetry of
      {rotate : int,
       reflect : bool,
       files : int,
       ranks : int,
       opponent : bool}

val identity : symmetry

val isIdentity : symmetry -> bool

(* Satisfies: transform s (invert s) = transform (invert s) s = identity *)
val invert : symmetry -> symmetry

(* Satisfies: transform (compose s1 s2) = transform s2 o transform s1 *)
val compose : symmetry -> symmetry -> symmetry

(* ------------------------------------------------------------------------- *)
(* Basic go board symmetries.                                                *)
(* ------------------------------------------------------------------------- *)

datatype reflectionAxis = Horizontal | Vertical | DiagonalSWNE | DiagonalNWSE

datatype rotationDirection = Anticlockwise | Clockwise

val translate : {files : int, ranks : int} -> symmetry

val rotateAround :
    {centre : Point.point, direction : rotationDirection,
     rightAngles : int} -> symmetry

val rotateBoard :
    Dimensions.dimensions ->
    {direction : rotationDirection, rightAngles : int} -> symmetry

val reflectAt : {point : Point.point, axis : reflectionAxis} -> symmetry

val reflectBoard : Dimensions.dimensions -> reflectionAxis -> symmetry

val opponent : symmetry

(* ------------------------------------------------------------------------- *)
(* Applying go board symmetries.                                             *)
(* ------------------------------------------------------------------------- *)

val transformPoint : symmetry -> Point.point -> Point.point

val transformSide : symmetry -> Side.side -> Side.side

(* ------------------------------------------------------------------------- *)
(* Parsing and pretty printing.                                              *)
(* ------------------------------------------------------------------------- *)

val toString : symmetry -> string

end
