(* ========================================================================= *)
(* GO BOARD SYMMETRY                                                         *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

structure Symmetry :> Symmetry =
struct

open Useful;

(* ------------------------------------------------------------------------- *)
(* The group of go board symmetries.                                         *)
(* ------------------------------------------------------------------------- *)

datatype symmetry =
    Symmetry of
      {rotate : int,
       reflect : bool,
       files : int,
       ranks : int,
       opponent : bool};

fun normalizeRotate n = n mod 4;

val identity =
    Symmetry
      {rotate = 0,
       reflect = false,
       files = 0,
       ranks = 0,
       opponent = false};

fun isIdentity sym =
    case sym of
      Symmetry
        {rotate = 0,
         reflect = false,
         files = 0,
         ranks = 0,
         opponent = false} => true
    | _ => false;

fun invert sym =
    if isIdentity sym then sym
    else
      let
        val Symmetry {rotate,reflect,files,ranks,opponent} = sym

        (* Swap the order of (files,ranks) and reflect *)
        val files = if reflect then ~files else files

        (* Swap the order of (files,ranks) and rotate *)
        val (files,ranks) =
            case rotate of
              0 => (files,ranks)
            | 1 => (ranks,~files)
            | 2 => (~files,~ranks)
            | 3 => (~ranks,files)
            | _ => raise Bug "bad rotate value"

        (* Swap the order of reflect and rotate *)
        val rotate = if reflect then ~rotate else rotate

        val rotate = normalizeRotate (~rotate)
        and files = ~files
        and ranks = ~ranks
      in
        Symmetry
          {rotate = rotate,
           reflect = reflect,
           files = files,
           ranks = ranks,
           opponent = opponent}
      end
(*GomiDebug
      handle Bug bug => raise Bug ("Symmetry.invert: " ^ bug);
*)

fun compose sym1 sym2 =
    if isIdentity sym1 then sym2
    else if isIdentity sym2 then sym1
    else
      let
        val Symmetry
              {rotate = rotate1,
               reflect = reflect1,
               files = files1,
               ranks = ranks1,
               opponent = opponent1} = sym1

        and Symmetry
              {rotate = rotate2,
               reflect = reflect2,
               files = files2,
               ranks = ranks2,
               opponent = opponent2} = sym2

        (* Swap the order of (files1,ranks1) and rotate2 *)
        val (files1,ranks1) =
            case rotate2 of
              0 => (files1,ranks1)
            | 1 => (~ranks1,files1)
            | 2 => (~files1,~ranks1)
            | 3 => (ranks1,~files1)
            | _ => raise Bug "bad rotate value"

        (* Swap the order of (files1,ranks1) and reflect2 *)
        val files1 = if reflect2 then ~files1 else files1

        (* Swap the order of reflect1 and rotate2 *)
        val rotate2 = if reflect1 then ~rotate2 else rotate2

        val rotate = normalizeRotate (rotate1 + rotate2)
        and reflect = reflect1 <> reflect2
        and files = files1 + files2
        and ranks = ranks1 + ranks2
        and opponent = opponent1 <> opponent2
      in
        Symmetry
          {rotate = rotate,
           reflect = reflect,
           files = files,
           ranks = ranks,
           opponent = opponent}
      end
(*GomiDebug
      handle Bug bug => raise Bug ("Symmetry.compose: " ^ bug);
*)

(* ------------------------------------------------------------------------- *)
(* Basic go board symmetries.                                                *)
(* ------------------------------------------------------------------------- *)

datatype reflectionAxis = Horizontal | Vertical | DiagonalSWNE | DiagonalNWSE;

datatype rotationDirection = Anticlockwise | Clockwise;

fun normalizeDirectionRotate Anticlockwise n = normalizeRotate n
  | normalizeDirectionRotate Clockwise n = normalizeRotate (~n);

fun translate {files,ranks} =
    Symmetry
      {rotate = 0,
       reflect = false,
       files = files,
       ranks = ranks,
       opponent = false};

fun rotateOrigin rotate =
    Symmetry
      {rotate = rotate,
       reflect = false,
       files = 0,
       ranks = 0,
       opponent = false};

fun rotateAround {centre,direction,rightAngles} =
    let
      val Point.Point {file,rank} = centre
      val rot = rotateOrigin (normalizeDirectionRotate direction rightAngles)
      and trans = translate {files = file, ranks = rank}
    in
      compose (invert trans) (compose rot trans)
    end;

fun rotateBoard dim {direction,rightAngles} =
    let
      val rotate = normalizeDirectionRotate direction rightAngles
      val _ = rotate = 2 orelse Dimensions.isSquare dim orelse
              raise Bug "can't rotate a rectangular board by 90 degrees"
      val {files,ranks} = dim
    in
      compose (rotateOrigin rotate)
      (case rotate of
         0 => identity
       | 1 => translate {files = ranks - 1, ranks = 0}
       | 2 => translate {files = files - 1, ranks = ranks - 1}
       | 3 => translate {files = 0, ranks = files - 1}
       | _ => raise Bug "bad rotate value")
    end
(*GomiDebug
    handle Bug bug => raise Bug ("Symmetry.rotateBoard: " ^ bug)
*)

val reflectOriginVertical =
    Symmetry
      {rotate = 0,
       reflect = true,
       files = 0,
       ranks = 0,
       opponent = false};

fun reflectOrigin axis =
    case axis of
      Horizontal =>
      compose (rotateOrigin 1) (compose reflectOriginVertical (rotateOrigin 3))
    | Vertical => reflectOriginVertical
    | DiagonalSWNE => compose (rotateOrigin 1) reflectOriginVertical
    | DiagonalNWSE => compose (rotateOrigin 3) reflectOriginVertical;

fun reflectAt {point,axis} =
    let
      val Point.Point {file,rank} = point
      val refl = reflectOrigin axis
      and trans = translate {files = file, ranks = rank}
    in
      compose (invert trans) (compose refl trans)
    end;

fun reflectBoard dim axis =
    let
      val {files,ranks} = dim
    in
      compose (reflectOrigin axis)
      (case axis of
         Horizontal => translate {files = 0, ranks = ranks - 1}
       | Vertical => translate {files = files - 1, ranks = 0}
       | DiagonalSWNE =>
         let
(*GomiDebug
           val _ = Dimensions.isSquare dim orelse
                   raise Bug "can't diagonally reflect a rectangular board"
*)
         in
           identity
         end
       | DiagonalNWSE =>
         let
(*GomiDebug
           val _ = Dimensions.isSquare dim orelse
                   raise Bug "can't diagonally reflect a rectangular board"
*)
         in
           translate {files = files - 1, ranks = ranks - 1}
         end)
    end
(*GomiDebug
    handle Bug bug => raise Bug ("Symmetry.reflectBoard: " ^ bug);
*)

val opponent =
    Symmetry
      {rotate = 0,
       reflect = false,
       files = 0,
       ranks = 0,
       opponent = true};

(* ------------------------------------------------------------------------- *)
(* Applying go board symmetries.                                             *)
(* ------------------------------------------------------------------------- *)

fun transformPoint sym point =
    let
      val Symmetry {rotate,reflect,files,ranks,...} = sym
      and Point.Point {file,rank} = point
      val file' = file
      and rank' = rank
      val (file',rank') =
          case rotate of
            0 => (file',rank')
          | 1 => (~rank',file')
          | 2 => (~file',~rank')
          | 3 => (rank',~file')
          | _ => raise Bug "bad rotate value"
      val file' = if reflect then ~file' else file'
      val file' = file' + files
      and rank' = rank' + ranks
    in
      if file' = file andalso rank' = rank then point
      else Point.Point {file = file', rank = rank'}
    end
(*GomiDebug
    handle Bug bug => raise Bug ("Symmetry.transformPoint: " ^ bug);
*)

fun transformSide (Symmetry {opponent,...}) side =
    if opponent then Side.opponent side else side;

(* ------------------------------------------------------------------------- *)
(* Parsing and pretty printing.                                              *)
(* ------------------------------------------------------------------------- *)

fun toString sym =
    let
      val Symmetry {rotate,reflect,files,ranks,opponent} = sym
      val l = []
      val l = if opponent then "swap sides" :: l else l
      val l = if files = 0 andalso ranks = 0 then l
              else ("add (" ^ Int.toString files ^ "," ^
                    Int.toString ranks ^ ")") :: l
      val l = if reflect then "reflect" :: l else l
      val l = if rotate = 0 then l
              else ("rotate " ^ Int.toString (90 * rotate)) :: l
    in
      if List.null l then "identity" else "<" ^ join ", " l ^ ">"
    end;

end
