(* ========================================================================= *)
(* GOMI BENCHMARKS                                                           *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

(* ------------------------------------------------------------------------- *)
(* Structures.                                                               *)
(* ------------------------------------------------------------------------- *)

open Useful;

(* ------------------------------------------------------------------------- *)
(* The program name.                                                         *)
(* ------------------------------------------------------------------------- *)

val PROGRAM = "benchmark";

(* ------------------------------------------------------------------------- *)
(* All possible benchmarks.                                                  *)
(* ------------------------------------------------------------------------- *)

datatype benchmark = Random | Undo | Knowledge;

val allBenchmarks = [Random,Undo,Knowledge];

fun benchmarkToString Random = "random"
  | benchmarkToString Undo = "undo"
  | benchmarkToString Knowledge = "knowledge";

fun benchmarkFromString "random" = Random
  | benchmarkFromString "undo" = Undo
  | benchmarkFromString "knowledge" = Knowledge
  | benchmarkFromString s = raise Bug ("benchmarkFromString: " ^ s);

val allBenchmarkStrings = List.map benchmarkToString allBenchmarks;

(* ------------------------------------------------------------------------- *)
(* Program options.                                                          *)
(* ------------------------------------------------------------------------- *)

val BENCHMARK : benchmark ref = ref Random;

val BENCHMARK_INITIAL : string option ref = ref NONE;

val BENCHMARK_SECONDS : int ref = ref 10;

local
  open Useful Options;
in
  val specialOptions =
      [{switches = ["--benchmark","-b"], arguments = ["BENCHMARK"],
        description = "the benchmark to run",
        processor =
          beginOpt
            (enumOpt allBenchmarkStrings endOpt)
            (fn _ => fn s => BENCHMARK := benchmarkFromString s)},
       {switches = ["--initial","-i"], arguments = ["file.sgf"],
        description = "the initial position to use",
        processor =
          beginOpt (stringOpt endOpt)
          (fn _ => fn s => BENCHMARK_INITIAL := SOME s)},
       {switches = ["--time","-t"], arguments = ["SECONDS"],
        description = "how long to run the benchmark",
        processor =
          beginOpt (intOpt (SOME 0, NONE) endOpt)
          (fn _ => fn n => BENCHMARK_SECONDS := n)}];
end;

val VERSION = "1.0";

val versionString = PROGRAM^" "^VERSION^"\n";

val programOptions =
    {name = PROGRAM,
     version = versionString,
     header = "usage: "^PROGRAM^" [option ...]\n" ^
              "Runs a Gomi benchmark test.\n",
     footer = "where BENCHMARK is one of {" ^
              join "|" allBenchmarkStrings ^ "}.\n",
     options = specialOptions @ Options.basicOptions};

fun succeed () = Options.succeed programOptions;
fun fail mesg = Options.fail programOptions mesg;
fun usage mesg = Options.usage programOptions mesg;

val (opts,work) =
    Options.processOptions programOptions (CommandLine.arguments ());

(* ------------------------------------------------------------------------- *)
(* Helper functions.                                                         *)
(* ------------------------------------------------------------------------- *)

fun startBenchmark s =
    TextIO.print
      ((if s = "" then ""
        else ("host: " ^ host () ^ ", time: " ^ time () ^
              ", date: " ^ date () ^ "\n")) ^
       "_____________________________________" ^
       "_____________________________________\n" ^
       (if s = "" then "" else s ^ "\n\n"));

fun endBenchmark () = startBenchmark "";

val haltBenchmark =
    let
      val start = Time.now ()
      val seconds = Time.fromReal (Real.fromInt (!BENCHMARK_SECONDS))
      val cutoff = Time.+ (start,seconds)
    in
      fn () =>
         let
           val now = Time.now ()
         in
           if Time.<= (now,cutoff) then NONE
           else SOME (Time.toReal (Time.- (now,start)))
         end
    end;

fun prettyIntToString i = Print.toString Print.ppPrettyInt i;

fun prettyRealToString r = prettyIntToString (Real.round r);

(* ------------------------------------------------------------------------- *)
(* Playing random games.                                                     *)
(* ------------------------------------------------------------------------- *)

local
  fun playRandomMove board empty toMove prefix =
      let
        val index = Portable.randomInt (prefix + 1)
      in
        if index = prefix then IMove.PASS
        else
          let
            val point = IIntSet.sub empty index
          in
            if IBoard.playSensibleStoneMove board toMove point then point
            else
              let
                val prefix = prefix - 1

                val () = IIntSet.swap empty index prefix
              in
                playRandomMove board empty toMove prefix
              end
          end
      end;

  fun playRandomGame board empty numMoves toMove passes =
      if passes >= 2 then numMoves
      else
        let
          val move = playRandomMove board empty toMove (IIntSet.size empty)
          val numMoves = numMoves + 1
          val toMove = Side.opponent toMove
          val passes = if move = IMove.PASS then passes + 1 else 0
        in
          playRandomGame board empty numMoves toMove passes
        end;
in
  fun randomBenchmark position =
      let
        val () = startBenchmark "UNIFORMLY RANDOM PLAY"

        val toMove = Position.toMove position

        val initialBoard = IBoard.fromBoard (Position.board position)

        val board = IBoard.clone initialBoard

        val empty = IBoard.empty board

        fun generate (m,g) =
            let
              val numMoves = playRandomGame board empty 0 toMove 0
              val (m,g) = (m + Real.fromInt numMoves, g + 1)
            in
              case haltBenchmark () of
                NONE => (IBoard.copy initialBoard board; generate (m,g))
              | SOME t => (m,g,numMoves,t)
            end

        val (m,g,numMoves,t) = generate (0.0,0)

        val dim = Position.dimensions position
        val () = TextIO.print ("Total seconds: " ^
                               prettyRealToString t ^ "\n")
        val () = TextIO.print ("Total " ^ Dimensions.toString dim ^
                               " games generated: " ^
                               prettyIntToString g ^ "\n")

        val g = Real.fromInt g
        val () = TextIO.print ("Games per second: " ^
                               prettyRealToString (g / t) ^ "\n")
        val () = TextIO.print ("Moves per second: " ^
                               prettyRealToString (m / t) ^ "\n")
        val () = TextIO.print ("Moves per game: " ^
                               prettyRealToString (m / g) ^ "\n")

        val () = TextIO.print ("\nFinal position of the last game " ^
                               "(which took " ^ prettyIntToString numMoves ^
                               " moves):\n" ^ IBoard.toString board ^ "\n")
      in
        ()
      end;
end;

(* ------------------------------------------------------------------------- *)
(* Testing the undoLastMove operation.                                       *)
(* ------------------------------------------------------------------------- *)

local
  fun playRandomMove board empty toMove prefix =
      let
        val index = Portable.randomInt (prefix + 1)
      in
        if index = prefix then IMove.PASS
        else
          let
            val point = IIntSet.sub empty index
          in
            if IBoard.playStoneMove board toMove point andalso
               let
                 val () = IBoard.undoLastMove board
               in
                 IBoard.playSensibleStoneMove board toMove point
               end
            then point
            else
              let
                val prefix = prefix - 1
                val () = IIntSet.swap empty index prefix
              in
                playRandomMove board empty toMove prefix
              end
          end
      end;

  fun playRandomGame board empty numMoves toMove passes =
      if passes >= 2 then numMoves
      else
        let
          val move = playRandomMove board empty toMove (IIntSet.size empty)
          val numMoves = numMoves + 1
          val toMove = Side.opponent toMove
          val passes = if move = IMove.PASS then passes + 1 else 0
        in
          playRandomGame board empty numMoves toMove passes
        end;
in
  fun undoBenchmark position =
      let
        val () = startBenchmark "UNIFORMLY RANDOM PLAY: MOVE, UNDO THEN REDO"

        val toMove = Position.toMove position

        val initialBoard = IBoard.fromBoard (Position.board position)

        val board = IBoard.clone initialBoard

        val empty = IBoard.empty board

        fun generate (m,g) =
            let
              val numMoves = playRandomGame board empty 0 toMove 0
              val (m,g) = (m + Real.fromInt numMoves, g + 1)
            in
              case haltBenchmark () of
                NONE => (IBoard.copy initialBoard board; generate (m,g))
              | SOME t => (m,g,numMoves,t)
            end

        val (m,g,numMoves,t) = generate (0.0,0)

        val dim = Position.dimensions position
        val () = TextIO.print ("Total seconds: " ^
                               prettyRealToString t ^ "\n")
        val () = TextIO.print ("Total " ^ Dimensions.toString dim ^
                               " games generated: " ^
                               prettyIntToString g ^ "\n")

        val g = Real.fromInt g
        val () = TextIO.print ("Games per second: " ^
                               prettyRealToString (g / t) ^ "\n")
        val () = TextIO.print ("Moves per second: " ^
                               prettyRealToString (m / t) ^ "\n")
        val () = TextIO.print ("Moves per game: " ^
                               prettyRealToString (m / g) ^ "\n")

        val () = TextIO.print ("\nFinal position of the last game " ^
                               "(which took " ^ prettyIntToString numMoves ^
                               " moves):\n" ^ IBoard.toString board ^ "\n")
      in
        ()
      end;
end;

(* ------------------------------------------------------------------------- *)
(* Building go knowledge.                                                    *)
(* ------------------------------------------------------------------------- *)

fun knowledgeBenchmark position =
    let
      val () = startBenchmark "BUILDING GO KNOWLEDGE BY PLAYING SAMPLE GAMES"

      val database = Database.new (Position.parameters position)

      val sample = ISample.new position database

      fun generate database (m,g) =
          let
            val database = ISample.sample sample database

            val numMoves = IStack.size (ISample.moves sample)

            val (m,g) = (m + Real.fromInt numMoves, g + 1)
          in
            case haltBenchmark () of
              NONE => generate database (m,g)
            | SOME t => (database,m,g,numMoves,t)
          end

      val (database,m,g,numMoves,t) = generate database (0.0,0)

      val dim = Position.dimensions position
      val () = TextIO.print ("Total seconds: " ^
                             prettyRealToString t ^ "\n")
      val () = TextIO.print ("Total " ^ Dimensions.toString dim ^
                             " games generated: " ^
                             prettyIntToString g ^ "\n")

      val g = Real.fromInt g
      val () = TextIO.print ("Games per second: " ^
                             prettyRealToString (g / t) ^ "\n")
      val () = TextIO.print ("Moves per second: " ^
                             prettyRealToString (m / t) ^ "\n")
      val () = TextIO.print ("Moves per game: " ^
                             prettyRealToString (m / g) ^ "\n")

      val board = IConfiguration.toBoard (ISample.configuration sample)
      val () = TextIO.print ("\nFinal position of the last game " ^
                             "(which took " ^ prettyIntToString numMoves ^
                             " moves):\n" ^ Board.toString board ^ "\n")

      val () = TextIO.print ("Database: " ^ Database.toString database ^ "\n");
    in
      ()
    end;

(* ------------------------------------------------------------------------- *)
(* Top level.                                                                *)
(* ------------------------------------------------------------------------- *)

val () =
let
  val position =
      case !BENCHMARK_INITIAL of
        SOME filename => Position.fromSgf {filename = filename}
      | NONE => Position.initialDefault

  val () =
      case !BENCHMARK of
        Random => randomBenchmark position
      | Undo => undoBenchmark position
      | Knowledge => knowledgeBenchmark position

  val () = endBenchmark ();
in
  succeed ()
end
handle Error s => die (PROGRAM^" failed:\n" ^ s)
     | Bug s => die ("BUG found in "^PROGRAM^" program:\n" ^ s);
