(* ========================================================================= *)
(* A GO PLAYING PROGRAM                                                      *)
(* Copyright (c) 2005 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

(* ------------------------------------------------------------------------- *)
(* Structures                                                                *)
(* ------------------------------------------------------------------------- *)

open Useful;

(* ------------------------------------------------------------------------- *)
(* The program name                                                          *)
(* ------------------------------------------------------------------------- *)

val PROGRAM = "gomi";

(* ------------------------------------------------------------------------- *)
(* Helper functions                                                          *)
(* ------------------------------------------------------------------------- *)

(* ------------------------------------------------------------------------- *)
(* Program options                                                           *)
(* ------------------------------------------------------------------------- *)

val RULES : Rules.rules ref = ref Rules.Chinese;

val KOMI : real option ref = ref NONE;

val BOARDSIZE : int ref = ref 9;

val ALL_RULES = List.map Rules.toString Rules.allRules;

local
  open Useful Options;
in
  val specialOptions =
      [{switches = ["--rules"], arguments = ["RULES"],
        description = "the rules of go in force",
        processor =
          beginOpt
            (enumOpt ALL_RULES endOpt)
            (fn _ => fn r => RULES := Rules.fromString r)},
       {switches = ["--komi"], arguments = ["{N|-}"],
        description = "the default komi",
        processor =
          beginOpt
            (optionOpt ("-", realOpt (NONE,NONE)) endOpt)
            (fn _ => fn k => KOMI := k)},
       {switches = ["--boardsize"], arguments = ["N"],
        description = "the default board size",
        processor =
          beginOpt (intOpt (SOME 2, NONE) endOpt)
          (fn _ => fn n => BOARDSIZE := n)}];
end;

val VERSION = "1.1";

val versionString = "Gomi "^VERSION^" (release 20180810)"^"\n";

val programOptions =
    {name = PROGRAM,
     version = versionString,
     header = "usage: "^PROGRAM^" [option ...]\n" ^
              "Plays a game of go using the GTP protocol.\n",
     footer = "where RULES is one of {" ^ join "|" ALL_RULES ^ "}.\n",
     options = specialOptions @ Options.basicOptions};

fun succeed () = Options.succeed programOptions;
fun fail mesg = Options.fail programOptions mesg;
fun usage mesg = Options.usage programOptions mesg;

val (opts,work) =
    Options.processOptions programOptions (CommandLine.arguments ());

(* ------------------------------------------------------------------------- *)
(* Top level                                                                 *)
(* ------------------------------------------------------------------------- *)

val () =
let
  val rules = !RULES

  val komi =
      case !KOMI of
        NONE => Rules.komi rules {handicap = 0}
      | SOME k => k

  val boardsize = !BOARDSIZE

  val gtp =
      GTP.initial
        {name = PROGRAM, version = VERSION,
         rules = rules, komi = komi, boardsize = boardsize}
  val () = GTP.loop {input = TextIO.stdIn, output = TextIO.stdOut} gtp
in
  succeed ()
end
handle Error s => die (PROGRAM^" failed:\n" ^ s)
     | Bug s => die ("BUG found in "^PROGRAM^" program:\n" ^ s);
