(* ========================================================================= *)
(* GOMI TESTS                                                                *)
(* Copyright (c) 2004 Joe Leslie-Hurd, distributed under the MIT license     *)
(* ========================================================================= *)

(* ------------------------------------------------------------------------- *)
(* Dummy versions of Moscow ML declarations to stop real compilers barfing.  *)
(* ------------------------------------------------------------------------- *)

(*mlton
val quotation = ref true;
val quietdec  = ref true;
val loadPath  = ref ([] : string list);
val load = fn (_ : string) => ();
*)

(*polyml
val quotation = ref true;
val quietdec  = ref true;
val loadPath  = ref ([] : string list);
val load = fn (_ : string) => ();
*)

(* ------------------------------------------------------------------------- *)
(* Load and open some useful modules                                         *)
(* ------------------------------------------------------------------------- *)

val () = loadPath := !loadPath @ ["../bin/mosml"];
val () = app load ["Options"];

open Useful;

val time = Portable.time;

(* ------------------------------------------------------------------------- *)
(* Helper functions                                                          *)
(* ------------------------------------------------------------------------- *)

fun SAY s =
    TextIO.print
      ("\n\n-------------------------------------" ^
       "-------------------------------------\n\n" ^ s ^ "\n\n");

val TEST =
    let
      val next = let val r = ref 0 in fn () => (r := !r + 1; !r) end
    in
      fn s => SAY ("Test " ^ Int.toString (next ()) ^ ": " ^ s)
    end;

(* ------------------------------------------------------------------------- *)
val () = TEST "probabilities";
(* ------------------------------------------------------------------------- *)

fun probToString s p =
    let
      val m = Probability.mean p
      and d = Probability.standardDeviation p
    in
      "mean(" ^ s ^ "): " ^ percentToString m ^ ", " ^
      "s.d.(" ^ s ^ "): " ^ percentToString d
    end;

val () =
    TextIO.print
      ("Probability.inverseConj:\n" ^ Probability.inverseConjInfo ^ "\n\n")

val x = Probability.uniformDistribution;
val () = TextIO.print (probToString "X" x ^ "\n");

val max_x_x = Probability.max x x;
val () = TextIO.print (probToString "max(X,X)" max_x_x ^ "\n");

val min_x_x = Probability.min x x;
val () = TextIO.print (probToString "min(X,X)" min_x_x ^ "\n");

val conj_x_x = Probability.conj x x;
val () = TextIO.print (probToString "X /\\ X" conj_x_x ^ "\n");

val disj_x_x = Probability.disj x x;
val () = TextIO.print (probToString "X \\/ X" disj_x_x ^ "\n");

(* ------------------------------------------------------------------------- *)
val () = TEST "the SGF file format";
(* ------------------------------------------------------------------------- *)

val sgf = GoSgf.read {filename = "example.sgf"};
val () =
    List.app (Position.sgfFoldVariations Position.initialDefault (K ())) sgf;
val () = GoSgf.write {collection = sgf, filename = "example.out.sgf"};

val position = Position.fromSgf {filename = "selfpin-velocity.sgf"};
val () = TextIO.print ("Position:\n" ^ Position.toString position);

(* ------------------------------------------------------------------------- *)
val () = TEST "undoing moves on imperative boards";
(* ------------------------------------------------------------------------- *)

fun testUndoMoves brd =
    let
      val board = IBoard.fromBoard brd
      val board' = IBoard.clone board

      fun testSideMove side move =
          let
            val legalMove =
                case IMove.destStoneMove move of
                  NONE =>
                  let
                    val () = IBoard.playPassMove board
                  in
                    true
                  end
                | SOME pt => IBoard.playStoneMove board side pt
          in
            if legalMove then
              let
                val () = IBoard.undoLastMove board
              in
                if IBoard.equal board board' then ()
                else raise Bug "not the same board"
              end
            else ()
          end
          handle Bug bug =>
            raise Bug ("side = " ^ Side.toString side ^ ", " ^
                       "move = " ^ IMove.toString move ^ ":\n" ^ bug)

      fun testMove move =
          let
            val () = testSideMove Side.Black move
            and () = testSideMove Side.White move
          in
            ()
          end

      val moves = IMove.PASS :: IIntSet.toList (IBoard.empty board)

      val () = app testMove moves

      val () = TextIO.print "Passed test\n";
    in
      ()
    end;

val () = testUndoMoves (Position.board position);

(* ------------------------------------------------------------------------- *)
val () = TEST "building a database of go knowledge";
(* ------------------------------------------------------------------------- *)

val () = TextIO.print "Initialize the database\n";
val database = time Database.new (Position.parameters position);
val () = TextIO.print ("Database: " ^ Database.toString database ^ "\n");

val () = TextIO.print "\nInitialize a game sampler\n";
val sample = time (ISample.new position) database;

val () = TextIO.print "\nPlay a sample game\n";
val database = time (ISample.sample sample) database;

val configuration = ISample.configuration sample;
val finalBoard = IConfiguration.toBoard configuration;
val () = TextIO.print ("\nFinal position:\n" ^ Board.toString finalBoard);

val numMoves = IStack.size (ISample.moves sample);
val () = TextIO.print ("\nMoves: " ^ Int.toString numMoves ^ "\n");

val points = IConfiguration.pointsScore configuration;
val score = Score.fromPoints (Position.komi position) points;
val () = TextIO.print ("\nScore: " ^ Score.toString score ^ "\n");

val () = TextIO.print ("Database: " ^ Database.toString database ^ "\n");
